/*
 * Lloyd Milligan (WA4EFS) Oct/Nov 2023
 * © CC attribution - https://creativecommons.org/licenses/by/3.0/us/
 *
 * Revisions:
 *             1.1  - Add screen-savers (Bubbles and Satellite glyph)
 * 
 *
 */

#include <Sgp4.h>  // Orbital dynamics
#include <Wire.h>
#include <LiquidCrystal_I2C.h>  // Frequency display
#include "SPI.h"
#include "Adafruit_GFX.h"  // Touch screen
#include "Adafruit_ILI9341.h"
#include "XPT2046_Touchscreen.h"
#include <SD.h>       // MicroSD card
#include <TimeLib.h>  // Real-time clock
#include <EEPROM.h>   // Implementation-specific parameters
#include "Data/DST.c"
#include <IRremote.hpp>  // RCA antenna rotator

const String VERSION = "1.1";

/*
  A4 - I2C interface - SDA
  A5 - I2C interface - SCK
*/

const int ROWS = 2;
const int COLS = 16;
LiquidCrystal_I2C lcd(0x27, COLS, ROWS);  // Instantiate (Tx and Rx Frequency displays)

#define TFT_CS 10
#define TFT_DC 9
#define TOUCH_CS 8

#define DEBUG_SWITCH 34  // Pin sensed on startup (DIP switch or jumper)

Adafruit_ILI9341 tft = Adafruit_ILI9341(TFT_CS, TFT_DC);
XPT2046_Touchscreen ts(TOUCH_CS);

// Basic TFT geometry
#define X_PIXELS 320
#define Y_PIXELS 240

const uint16_t button_width = 280;  // Pixels
const uint16_t button_height = 20;
const uint16_t button_half_height = 10;
const uint16_t button_x0 = 20;
const uint16_t button_y0 = 10;
const uint16_t button_radius = 5;
const uint16_t normalTextSize = 2;
const uint16_t status_line_x = 10;
const uint16_t status_line_y = Y_PIXELS - button_height;
const uint16_t touch_display_x = status_line_x;
const uint16_t time_display_x = 200;
const uint16_t load_button_x0 = 15;
const uint16_t load_button_width = 290;
const uint16_t di_buttons_width = 140;
const uint16_t di_buttons_space = 10;
const uint16_t tri_buttons_width = 95;  // 1/3 load button width minus space between
const uint16_t tri_buttons_space = 5;
const uint16_t hexa_buttons_width = 44;  // 1/6 load button width minus space between
const uint16_t hexa_buttons_space = 5;
const uint16_t sat_data_x0 = 20;
// Vertical positioning of text and buttons
const int returnButtonYindex = 9;     // Multiple screens
const int txDopplerSwitchYindex = 5;  // Edit value to change vertical position of switch
const int rxDopplerSwitchYindex = 6;
const int lockUnplugSwitchesYindex = 8;
const int predictedPassMaxElvYindex = 4;
const int predictedPassDateTimeYindex = 5;
const int passAzRangeYindex = 2;
const int passDopplersCaptionYindex = 3;
const int passDopplersDataYindex = 4;
const int cancelSaveButtonsYindex = 4;
const int vfoCalibrationYindex = 6;
const int vfoCalibrationSubButtonXindex = 2;
const int vfoCalibrationAddButtonXindex = 5;

#define ROTARY_ENCODER_A 2  // Rotary encoder A-phase (white wire) (Interrupt)
#define ROTARY_ENCODER_B 3  // Rotary encoder B-phase (green wire) (Interrupt)

volatile long rawCount = 0;  // Rotary encoder raw data (converts to VFO frequency)
long lastRawCount = 0;       // Detect change in raw count

// Standard colors
#define BLACK 0x0000
#define BLUE 0x001F
#define RED 0xF800
#define GREEN 0x07E0
#define CYAN 0x07FF
#define MAGENTA 0xF81F
#define YELLOW 0xFFE0
#define WHITE 0xFFFF

// Additional colors (RGB)
// https://www.w3schools.com/colors/colors_picker.asp
uint8_t LT_GREY[] = { 224, 224, 224 };
uint8_t TAN[] = { 255, 236, 179 };
uint8_t LT_BROWN[] = { 255, 224, 179 };
uint8_t BROWN[] = { 153, 92, 0 };

// Color aliases
#define TFT_BG YELLOW
#define TFT_TXT BLUE  // Text color 1
#define TFT_ALT RED   // Text color 2

uint16_t mainscreen_background_color = BLACK;
uint16_t toolscreen_background_color = mainscreen_background_color;
uint16_t highlight_text_color = TFT_ALT;
uint16_t highlight_background_color = TFT_BG;
uint16_t satData_text_color = WHITE;
uint16_t satData_background_color = BLACK;
uint16_t status_bar_text_color = TFT_TXT;
uint16_t status_bar_text_size = normalTextSize;
uint16_t vfoCalibrationButtonsTextColor = BLACK;
uint16_t vfoCalibrationButtonsBackgroundColor = TFT_BG;


const unsigned long C = 299792458;  // Speed of light in vacuum (meters per second)

// Time constants
const unsigned long MILLISEC = 1;
const unsigned long MOMENT = 10;
const unsigned long DEBOUNCE_TOUCH = 100;
const unsigned long HALFSEC = 500;
const unsigned long ONESEC = 1000;
const unsigned long TWOSEC = 2000;
const unsigned long FIVESEC = 5000;
const unsigned long TENSEC = 10000;
const unsigned long HALFMIN = 30000;
const unsigned long ONEMIN = 60000;

// Next: whole number of seconds (generally one second)
const int SGP4_UPDATE_SECONDS = 1;
// Same in milliseconds
const unsigned long SGP4_UPDATE_INTERVAL = SGP4_UPDATE_SECONDS * ONESEC;
const unsigned long LCD_TIMEOUT = HALFMIN;

int secondsCounter = 0;  // Pace updating of satellite info
unsigned long lastLCDupdate = 0;
unsigned long lastStatusUpdate = 0;

const int RX_BUFFER_DIM = 80;
char serRxBuffer[RX_BUFFER_DIM];
int rxNdx = 0;

#define antInterface Serial5   // Antenna rotator interface
uint32_t antBAUD = 9600;

#define rigInterface0 Serial6  // Transmitter interface (BAUD is also a station file parameter)
uint32_t rig0BAUD = 9600;      // See note below

#define rigInterface1 Serial7  // Receiver interface - To do: Substitute parameter value
uint32_t rig1BAUD = 9600;      // in initRadioInterfaces()

String testAntInterface = "Hello, antenna!";
String loopbackTest = " Loopback Test;";

byte EOC = 0x3B;  // Yaesu CAT end-of-command
byte EOM = 0xFD;  // Icom CI-V end-of message
boolean rigCommandReceived = false;
const char SP = ' ';
const char ZERO = '0';
const char EEDLM = '@';  // EEPROM field delimiter / Data verifier
const byte MINUS = 45;   // Negative number (or hyphen)
const char POINT = '.';  // Decimal point

String CIV_Frequency = "";  // BCD format 10-digits in 5 bytes, last digit being
                            // the constant 0 for 1000's MHz. See Icom specification.
                            // Application does not support the 1240 MHz (23 cm) band

// MicroSD
Sd2Card card;
SdVolume volume;
SdFile root;

const int chipSelect = BUILTIN_SDCARD;
boolean SD_card_enabled = false;
boolean tleFileFound = false;
unsigned int tleCount = 0;

const int MAX_FILES = 20;
String fileList[MAX_FILES];
unsigned int fileCount = 0;
File selectedFile;

const int MAX_LINES = 1000;
String textFileData[MAX_LINES];

const int MAX_SATS = 8;  // Temporary limit until screen paging implemented
String freqList[MAX_SATS];
int satCount = 0;

byte TAB = 9;    // Frequencies file field delimiter
byte EOL = 13;   // Text files record delimiter
byte CR = 0x0D;  // Same as EOL (return) - Interface context
byte LF = 0x0A;  // Linefeed

String satNames[MAX_SATS];  // For convenience
int satTLEindex[MAX_SATS];
int selectedSatellite = -1;      // Zero-based display list order # 0, 1, ...
int lastSelectedSatellite = -1;  // Remember the satellite that was selected last
int loadButtonIndex = returnButtonYindex;

const int MAX_TLE = 200;
#define TLEdata textFileData  // Contextual alias

const int LINES = 2;
String currentTLE[LINES];

const int MAX_PARAMS = 20;  // Station file parameters
String stationParams[MAX_PARAMS];

String sCallsign = "";  // Breakdown of station parameters
float fLat = 0.;
float fLon = 0.;
float fAlt = 0.;
float fTZ = 0.;       // Timezone (Local standard time - UT delta)
int DST_hours;        // Numeric 1 or 0
boolean DST = false;  // Station parameter - not independently computed in sketch
boolean xmtInterface = false;
boolean rcvInterface = false;
String xmtType = "";  // See example station.txt file for definition
String rcvType = "";
uint32_t xmtBAUD = 0;
uint32_t rcvBAUD = 0;
String tuningPreference = "";  // Ditto
String DopplerPreference = "";

#define TIME_HEADER "T"  // For serial time sync message

uint16_t ts_x, ts_y;  // touch coordinates
uint8_t ts_z;         // touch pressure, not used
boolean touch_being_processed = false;
int screenID = 0;

String selectedSatelliteFrequencies = "";
unsigned long transmitFrequencyLB;
unsigned long transmitFrequencyUB;
unsigned long receiveFrequencyLB;  // Read as 'left bound' rather than 'lower bound' etc.
unsigned long receiveFrequencyUB;
unsigned long transmitFrequencyVFO;
unsigned long receiveFrequencyVFO;
boolean transmitFrequencyInverted = false;
boolean receiveFrequencyInverted = false;
unsigned long transmitDopplerOffset;
unsigned long receiveDopplerOffset;
unsigned long transmitFrequencyDopplerVFO;
unsigned long receiveFrequencyDopplerVFO;
boolean txDopplerCorrectionEnabled = false;  // Toggle switch - Satellite details screen
boolean rxDopplerCorrectionEnabled = false;
boolean pass_in_progress = false;
int defaultVFOstep = 5;        // 2 KHz per 360° revolution of tuning dial
int VFOstep = defaultVFOstep;  // Will be user-configurable in settings

Sgp4 sat;
unsigned long unixTime;  // SGP4 methods want unsigned long type

float satCurrentRange = 0.;  // Distance of satellite from station in kilometers
float satLastRange = 0.;
float satRangeDelta;
float satCurrentElv;  // Determines whether pass is in progress or not.
float satVelocity;
String satMaxElv;  // Fixed parameter of pass (aka overpass)
String satAz;
unsigned long satCurrentRangeTimestamp = 0;
unsigned long satLastRangeTimestamp = 0;
long satRangeTimeDelta;

// For 16 x 2 LCD smooth display function
String lastLCD[ROWS] = { "                ", "                " };

boolean context_free_tuning = true;      // Enable tuning whether satellite is above or below horizon
boolean context_free_send_to_Rx = true;  // Send Rx tuning data to receiver
boolean context_free_send_to_Tx = true;  // Send Tx tuning data to transmitter
unsigned long lastReceiveFrequencyVFO;   // Only send tuning data when frequency has changed by more than ->
long min_VFO_delta = 100;                // Pace context-free frequency updates (Hz)
String lastCIV_Frequency = "";           // Filter duplicate send of same CIV_Frequency

boolean enableDateTimeEdit = false;  // Tools (Screen 3)
int deltaYear = 0;                   // All deltas are signed
int deltaMonth = 0;                  // RTC value + delta = adjusted value
int deltaDay = 0;
int deltaHour = 0;  // 24 hour format
int deltaMinute = 0;
int deltaSecond = 0;

// From TimeLib.h
#define LEAP_YEAR(Y) (((1970 + (Y)) > 0) && !((1970 + (Y)) % 4) && (((1970 + (Y)) % 100) || !((1970 + (Y)) % 400)))
static const uint8_t monthDays[] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

const int HOST_TIME_DELTA = 1;  // ± Seconds to add when setting time from host
int lastTick = -1;              // Sync per second status bar time updating to RTC
int lastSgp4Tick = -1;          // Sync satellite info updating to RTC

boolean passPhaseAdjustmentEnabled = false;  // Show/hide buttons etc.
int passPhaseDeltaSecond = 0;                // Phase-shift the pass vis-a-vis RTC time

boolean txInterfaceDisconnect = false;        // Equivalent disconnecting transmitter interface cable
boolean rxInterfaceDisconnect = false;        // Ditto for receive interface
boolean antInterfaceDisconnect = false;       // Ditto for antenna interface
boolean txFrequencyLock = false;              // Lock uncorrected transmit frequency
boolean rxFrequencyLock = false;              // Ditto for receive interface
boolean connectToggleButtonPressed = true;    // Prevent repainting when not changed
int statusMessageNumber = -1;                 // Facilitate selection of contextual status message
const int ANT_INTERFACE_UPDATE_INTERVAL = 5;  // (seconds) Pace sending antenna data to rotator
int rotorInterfaceType = 1;                   // Enum 0 = Custom Yaesu-like Mxxx message
                                              //      1 = EASYCOMM-1 format AZx <SP> ELx <CR>

boolean calibrationToolEnabled = false;  // Calibration tool display on Tools page
boolean txCalibrateEnabled = false;      // Applies linear calibration adjustment when sending Tx Doppler
boolean rxCalibrateEnabled = false;      // Applies linear calibration adjustment when sending Rx Doppler
float txSlope = 0.;                      // Multiplier for Tx adjustment
float rxSlope = 0.;                      // Multiplier for Rx adjustment
int txIntercept = 0;                     // Add (Hz) to Tx Doppler correction
int rxIntercept = 0;                     // Add (Hz) to Rx Doppler correction
const int vfoCalibrationDelta = 50;      // Step value in Hz to add/subtract for VFO calibration
boolean invertTransmitDoppler = false;   // Toggle using soft test button T05 (Deprecated)

const unsigned int EE_BUFFER_DIM = 0x10BC;
byte eeBuffer[EE_BUFFER_DIM];  // Convenience

const int testButtonsYindex = 8;
const int MAX_TEST_CONDITIONS = 6;
boolean test_condition[MAX_TEST_CONDITIONS] = { false, false, false, false, false, false };
boolean displayTestButtons = false;
boolean debugSimulatePassInProgress = false;  // Toggle using soft test button T00
boolean testAzSerialInterface = false;

const int MAX_CANNED_STATUS_MESSAGES = 3;  // Expand as needed
String statusMessage[MAX_CANNED_STATUS_MESSAGES] = { "Tx Doppler Inv", "Simulated pass", "AzSerial Test" };

// EEPROM field start indexes
const unsigned int EE_START = 0;       // If EEPROM has valid data, position [EE_START] will contain EEDLM
const unsigned int EE_INTERCEPTS = 1;  // TX (6 bytes including sign) + EEDLM + RX (6 bytes) + EEDLM

// Screen saver
const unsigned long SCREEN_TIMEOUT = 60000;   // Start screensaver after this time in main screen or Tools screen
const uint16_t ssBkgColor = BLACK;
unsigned long lastTouched = millis();
boolean screenSaver = false;                  // True while screensaver is ON (active)
unsigned int screenSaverType = 1;             // 0 = Bubbles, 1 = Satellite glyph

void setup() {
  Serial.begin(115200);                 // Debug
  pinMode(DEBUG_SWITCH, INPUT_PULLUP);  // Jumper to SPST DIP switch

  lcd.begin(COLS, ROWS);
  lcd.init();
  lcd.noBacklight();

  tft.begin();  // Init TFT LCD screen
  tft.setRotation(1);
  tft.fillScreen(ILI9341_BLACK);
  ts.begin();  // Init Touch
  ts.setRotation(3);

  testLCD();
  tftSplash();
  myDelay(FIVESEC);
  tftOff();

  pinMode(ROTARY_ENCODER_A, INPUT_PULLUP);
  pinMode(ROTARY_ENCODER_B, INPUT_PULLUP);
  attachInterrupt(digitalPinToInterrupt(ROTARY_ENCODER_A), isrA, RISING);
  attachInterrupt(digitalPinToInterrupt(ROTARY_ENCODER_B), isrB, RISING);

  initRadioInterfaces();
  zeroRxBuffer();

  delay(TWOSEC);

  if (card.init(SPI_HALF_SPEED, chipSelect)) {
    SD_card_enabled = true;  // Formatted as FAT32
    loadFileList();
    if (isTLEfile() == true)  // Debug
      tleFileFound = true;
  }

  if (!tleFileFound) {  // Panic message
    displayLCD("", "*TLEs not found!");
    delay(TWOSEC);
    lcd.noBacklight();
  }

  getTimeFromHost();  // Time can also be set or adjusted in Tools screen
  initClock();

  readMicroSD();
  initConvenienceArrays();
  initTxRxVFOcalibration();
  eeStartupRestore();
  antInterface.begin(antBAUD, SERIAL_8N1);
  loadMainScreen();

  processDebugSwitch();
  initScreenSaver();                          // Arduino TFT bubbles example

  delay(ONESEC);
}

void loop() {
  long VFO_delta;

  if (ts.touched() && !touch_being_processed)
    tftProcessTouch();
/*
  rigInterface0Listen();  // For Icom OK or NG message - Debug
  if (rigCommandReceived) {
    processReceivedCivMessage();
    zeroRxBuffer();
  }
*/
  if (rawCount != lastRawCount) {
    if (selectedSatellite >= 0 || (screenID == 1 && context_free_tuning && lastSelectedSatellite >= 0)) {
      updateVFO();
      displayVFO();

      if (context_free_send_to_Rx && !rxDopplerCorrectionEnabled) {
        VFO_delta = lastReceiveFrequencyVFO - receiveFrequencyVFO;
        if (VFO_delta < 0)
          VFO_delta = -VFO_delta;
        if (VFO_delta > min_VFO_delta) {
          sendReceiveFrequencyUpdate();
          lastReceiveFrequencyVFO = receiveFrequencyVFO;
          if (context_free_send_to_Tx && !txDopplerCorrectionEnabled) {
            sendTransmitFrequencyUpdate();
          }
        }
      }
    }
    lastRawCount = rawCount;
  } else {
    if (millis() - lastLCDupdate > LCD_TIMEOUT)
      lcd.noBacklight();
  }

  if (second() != lastTick && !screenSaver) { // Do once per second
    lastTick = second();
    tftDisplayStatusbar();
    tftDisplayStatusbar(statusMessageText());
    if (screenID == 3) {
      tftDisplayEditableDateTimeRow(1, WHITE, 2);
    }
    if (testAzSerialInterface)
      testSendAz();
    lastStatusUpdate = millis();
  }

  if (screenID == 2) {
    updateSatelliteInfo();
  }

  if (screenSaver) {
    if (screenSaverType == 0)
      oneBubblesScreenSaverIteration();
    else if (screenSaverType == 1)
      oneSatScreenSaverIteration(TWOSEC);
    delay(MILLISEC);
  }
  else if (screensaverOn())
    screenSaver = true;    

}

float fDoppler(unsigned long v, float f) {
  return ((float)C / (float)(C + v)) * f;
}

// -------------------------------------LCD----------------------------------------

void displayLCD(String line1, String line2) {
  smoothDisplayLCD(line1, line2);
  return;
}

void smoothDisplayLCD(String line1, String line2) {
  unsigned int i;
  unsigned int L1, L2;
  L1 = line1.length();
  L2 = line2.length();
  for (i = 0; i < L1; i++)
    if (line1.charAt(i) != lastLCD[0].charAt(i)) {
      lcd.setCursor(i, 0);
      lcd.print(line1.charAt(i));
    }
  for (i = L1; i < COLS; i++) {
    lcd.setCursor(i, 0);
    lcd.print(SP);
  }
  for (i = 0; i < L2; i++)
    if (line2.charAt(i) != lastLCD[1].charAt(i)) {
      lcd.setCursor(i, 1);
      lcd.print(line2.charAt(i));
    }
  for (i = L2; i < COLS; i++) {
    lcd.setCursor(i, 1);
    lcd.print(SP);
  }
  lastLCD[0] = line1;
  lastLCD[1] = line2;
  lcd.backlight();
  lastLCDupdate = millis();
}

void testLCD() {
  for (int i = 0; i < 1; i++) {
    displayLCD("  W A 4 E F S", " This is a test");  // Substitute personalized text
    delay(TWOSEC);                                   // Extend duration of splash screen
    lcd.noBacklight();
    lcd.clear();
    delay(ONESEC);
  }
}

// -------------------------------------VFO----------------------------------------

void updateVFO() {  // VFO's - both, if enabled
  int delta = rawCount - lastRawCount;
  if (delta == 0)  // Redundant check
    return;
  delta *= VFOstep;
  // Note: The symbols LB and UB refer to the left and right satellite frequency bounds
  //       LB may be either less than or greater than UB
  //       Typically for inverting linear transponders, the receive LB will be > its UB
  //       The following assumes that transmit LB is less than transmit UB and that
  //       receive LB may be either less than or greater than receive UB
  if (transmitFrequencyVFO > 0 && (transmitFrequencyLB != transmitFrequencyUB) && !txFrequencyLock)
    transmitFrequencyVFO += delta;
  if (receiveFrequencyVFO > 0 && (receiveFrequencyLB != receiveFrequencyUB) && !rxFrequencyLock) {
    if (receiveFrequencyLB < receiveFrequencyUB)
      receiveFrequencyVFO += delta;
    else
      receiveFrequencyVFO -= delta;
  }
  // Comment-out the following to allow out-of-band tuning
  if (transmitFrequencyVFO > transmitFrequencyUB)
    transmitFrequencyVFO = transmitFrequencyUB;
  else if (transmitFrequencyVFO < transmitFrequencyLB)
    transmitFrequencyVFO = transmitFrequencyLB;

  if (receiveFrequencyLB < receiveFrequencyUB) {
    if (receiveFrequencyVFO > receiveFrequencyUB)
      receiveFrequencyVFO = receiveFrequencyUB;
    else if (receiveFrequencyVFO < receiveFrequencyLB)
      receiveFrequencyVFO = receiveFrequencyLB;
  } else if (receiveFrequencyLB > receiveFrequencyUB) {
    if (receiveFrequencyVFO < receiveFrequencyUB)
      receiveFrequencyVFO = receiveFrequencyUB;
    else if (receiveFrequencyVFO > receiveFrequencyLB)
      receiveFrequencyVFO = receiveFrequencyLB;
  }
}

void resyncTransmitVFO() {
  unsigned long rOffset;
  if (receiveFrequencyLB < receiveFrequencyUB)
    rOffset = receiveFrequencyVFO - receiveFrequencyLB;
  else if (receiveFrequencyLB > receiveFrequencyUB)
    rOffset = receiveFrequencyLB - receiveFrequencyVFO;
  else
    return;  // Nothing to synch with
  transmitFrequencyVFO = transmitFrequencyLB + rOffset;
  displayVFO();
}

void resyncReceiveVFO() {
  unsigned long tOffset;
  if (transmitFrequencyUB > transmitFrequencyLB)
    tOffset = transmitFrequencyVFO - transmitFrequencyLB;
  else
    return;  // Nothing to synch with
  if (receiveFrequencyLB < receiveFrequencyUB)
    receiveFrequencyVFO = receiveFrequencyLB + tOffset;
  else
    receiveFrequencyVFO = receiveFrequencyLB - tOffset;
  displayVFO();
}

void displayVFO() {
  String s = "", s1 = "", s2 = "";
  if (transmitFrequencyVFO > 0) {
    s = String(transmitFrequencyVFO);
    if (s.length() == 9) {
      s1 = "Tx: ";
      s1.concat(s.substring(0, 3));
      s1.concat(POINT);
      s1.concat(s.substring(3, 6));
      s1.concat(POINT);
      s1.concat(s.substring(6));
    } else {
      s1 = "Tx: ";
      s1.concat(s);
      s1.concat(" Hz");
    }
  }
  if (receiveFrequencyVFO > 0) {
    s = String(receiveFrequencyVFO);
    if (s.length() == 9) {
      s2 = "Rx: ";
      s2.concat(s.substring(0, 3));
      s2.concat(POINT);
      s2.concat(s.substring(3, 6));
      s2.concat(POINT);
      s2.concat(s.substring(6));
    } else {
      s1 = "Tx: ";
      s1.concat(s);
      s1.concat(" Hz");
    }
  }
  displayLCD(s1, s2);
}

// Experimental frequency correction (transmitter or receiver VFO linear offset)
// For now, additive only (slope = 0)
void initTxRxVFOcalibration() {
  // Startup defaults
  txCalibrateEnabled = true;
  rxCalibrateEnabled = false;
  // Retrieve last calibration values from EEPROM here
}

void toggleTxRxVFOcalibration() {
  txCalibrateEnabled = !txCalibrateEnabled;
  rxCalibrateEnabled = !rxCalibrateEnabled;
}

// -------------------------------------TFT----------------------------------------

void tftSplash() {                            // Customize as desired
  screenID = 0;
  tft.fillScreen(TFT_BG);
  tft.setTextColor(TFT_ALT);
  tft.setTextSize(normalTextSize);
  tft.setCursor(40, 80);
  tft.println("   Ham Sat Assist");
  tft.setCursor(35, 100);
  tft.println("    version " + String(VERSION));
  tft.setTextColor(TFT_TXT);
  tft.setCursor(30, 140);
  tft.println("https://www.lloydm.net");
}

void tftDrawSatelliteGlyph(uint16_t x0, uint16_t y0, uint16_t lineColor, uint16_t fillColor) {
  const uint16_t WING_WIDTH = 32;
  const uint16_t WING_HEIGHT = 14;
  const uint16_t WING_PANEL_WIDTH_HEIGHT = 4;
  const uint16_t BODY_WIDTH = 16;
  const uint16_t BODY_HEIGHT = 40;
  const uint16_t NOSE_DEPTH = 8;
  const uint16_t WING_BODY_GAP = 2;
  const uint16_t PX1 = 1;
  const uint16_t PX2 = 2;
  const uint16_t PX4 = 4;
  const uint16_t topArcRadius = BODY_WIDTH/2 - 1;
  const uint16_t BTM_ARCS_GAP = 4;
  const uint16_t btmArcRadius1 = 10;
  // Convenience variables
  uint16_t bodyX = x0 + WING_WIDTH + WING_BODY_GAP;
  uint16_t rightWingX = x0 + WING_WIDTH + BODY_WIDTH + 2*WING_BODY_GAP;
  uint16_t arcX = x0 + WING_WIDTH + WING_BODY_GAP + BODY_WIDTH/2;
  uint16_t topArcY = y0;
  uint16_t btmArcY1 = y0 + BODY_HEIGHT;
  // Triangle points
  uint16_t leftX = arcX - (BODY_WIDTH + WING_BODY_GAP);
  uint16_t rightX = arcX + (BODY_WIDTH + WING_BODY_GAP);
  uint16_t btmY = btmArcY1 + 2*(btmArcRadius1 + BTM_ARCS_GAP);

  // Curved nose
  tft.drawCircle(arcX, topArcY, topArcRadius, lineColor);
  tft.fillRect(x0 + WING_WIDTH + WING_BODY_GAP, y0, BODY_WIDTH, BODY_WIDTH, ssBkgColor);
  // Bottom arcs
  tft.drawCircle(arcX, btmArcY1, btmArcRadius1, lineColor);
  tft.drawCircle(arcX, btmArcY1 + BTM_ARCS_GAP, btmArcRadius1  + BTM_ARCS_GAP/2, lineColor);
  tft.drawCircle(arcX, btmArcY1 + 2*BTM_ARCS_GAP, btmArcRadius1  + BTM_ARCS_GAP, lineColor);
  // Cut arcs from circles (Erasesures)
  tft.fillRect(x0, y0, BODY_WIDTH + 2*WING_WIDTH + 2*WING_BODY_GAP, BODY_HEIGHT, ssBkgColor);
  tft.fillTriangle(leftX, btmArcY1, arcX, btmArcY1, leftX, btmY, ssBkgColor);
  tft.fillTriangle(arcX, btmArcY1, rightX, btmY, rightX, btmArcY1, ssBkgColor);
  // Left wing
  tft.drawRect(x0, y0+NOSE_DEPTH, WING_WIDTH, WING_HEIGHT, lineColor);
  for (unsigned int i=1; i<6; i++)
    tft.fillRect(x0 + i*WING_PANEL_WIDTH_HEIGHT+i - PX1, y0 + NOSE_DEPTH + PX2, WING_PANEL_WIDTH_HEIGHT, WING_PANEL_WIDTH_HEIGHT, fillColor);
  for (unsigned int i=1; i<6; i++)
    tft.fillRect(x0 + i*WING_PANEL_WIDTH_HEIGHT+i - PX1, y0 + NOSE_DEPTH + WING_PANEL_WIDTH_HEIGHT + PX4, WING_PANEL_WIDTH_HEIGHT, WING_PANEL_WIDTH_HEIGHT, fillColor);
  // Body
  tft.drawRect(bodyX, y0, BODY_WIDTH, BODY_HEIGHT, lineColor);
  // Right wing
  tft.drawRect(rightWingX, y0+NOSE_DEPTH, WING_WIDTH, WING_HEIGHT, lineColor);
  for (unsigned int i=1; i<6; i++)
    tft.fillRect(rightWingX + i*WING_PANEL_WIDTH_HEIGHT+i - PX1, y0 + NOSE_DEPTH + PX2, WING_PANEL_WIDTH_HEIGHT, WING_PANEL_WIDTH_HEIGHT, fillColor);
  for (unsigned int i=1; i<6; i++)
    tft.fillRect(rightWingX + i*WING_PANEL_WIDTH_HEIGHT+i - PX1, y0 + NOSE_DEPTH + WING_PANEL_WIDTH_HEIGHT + PX4, WING_PANEL_WIDTH_HEIGHT, WING_PANEL_WIDTH_HEIGHT, fillColor);
}

void tftOff() {
  tft.fillScreen(BLACK);
}

void tftDisplayButton(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t fg, uint16_t bg, String lbl, uint16_t sz) {
  tft.fillRoundRect(x, y, w, h, button_radius, bg);
  tft.setTextColor(fg);
  tft.setTextSize(sz);
  tft.setCursor(x + 5, y + 2);
  tft.print(lbl);
  tft.drawRoundRect(x, y, w, h, button_radius, BLACK);
  return;
}

void tftDisplayMsg(String msg) {
  tft.fillScreen(TFT_BG);
  tft.setCursor(0, 200);
  tft.setTextColor(RED);
  tft.setTextSize(1);
  tft.println("Debug: " + msg);
}

void tftDisplaySatelliteNames() {
  for (int i = 0; i < satCount; i++) {
    if (satNames[i] == "")
      return;
    tftDisplayButton(button_x0, button_y0 + i * button_height, button_width, button_height, MAGENTA, uColor(LT_GREY), satNames[i], normalTextSize);
  }
}

void tftHighlightSatelliteName(int iSat) {
  if (iSat < 0)
    return;
  tftDisplaySatelliteNames();  // Clear last selected name
  tftDisplayButton(button_x0, button_y0 + iSat * button_height, button_width, button_height, highlight_text_color, highlight_background_color, satNames[iSat], normalTextSize);
}

void tftDisplayStatusbar() {
  tft.fillRect(0, status_line_y - 1, X_PIXELS, button_height, uColor(LT_GREY));
  tft.setTextColor(status_bar_text_color);  // To do: Parameterize status color and size
  tft.setTextSize(status_bar_text_size);
  tft.setCursor(time_display_x, status_line_y);
  tft.println(sTime());
}

void tftDisplayStatusbar(String status) {
  // Parameter status should leave room for time
  if (status.length() > 15)
    status = status.substring(0, 15);
  tft.fillRect(0, status_line_y - 1, X_PIXELS, button_height, uColor(LT_GREY));
  tft.setTextColor(status_bar_text_color);
  tft.setTextSize(status_bar_text_size);
  tft.setCursor(status_line_x, status_line_y);
  tft.println(status);
  tft.setCursor(time_display_x, status_line_y);
  tft.println(sTime());
}

void tftDisplayLoadSatelliteButton() {
  tft.fillRoundRect(load_button_x0, button_y0 + (MAX_SATS + 1) * button_height, load_button_width, button_height, button_radius, highlight_background_color);
  tftDisplayButton(load_button_x0, button_y0 + (MAX_SATS + 1) * button_height, load_button_width, button_height, highlight_text_color, highlight_background_color, "Load selected satellite", normalTextSize);
}

void tftDisplayGenericLeftButton(String caption) {
  tftDisplayButton(load_button_x0, button_y0 + (MAX_SATS + 1) * button_height, tri_buttons_width, button_height, highlight_text_color, highlight_background_color, caption, normalTextSize);
}

void tftDisplayGenericMiddleButton(String caption) {
  tftDisplayButton(load_button_x0 + tri_buttons_width + tri_buttons_space, button_y0 + (MAX_SATS + 1) * button_height, tri_buttons_width, button_height, highlight_text_color, highlight_background_color, caption, normalTextSize);
}

void tftDisplayGenericRightButton(String caption) {
  tftDisplayButton(load_button_x0 + 2 * tri_buttons_width + 2 * tri_buttons_space, button_y0 + (MAX_SATS + 1) * button_height, tri_buttons_width, button_height, highlight_text_color, highlight_background_color, caption, normalTextSize);
}

void tftDisplayMediumLengthLeftButton(int yNdx, String caption) {
  tftDisplayButton(load_button_x0, button_y0 + yNdx * button_height, di_buttons_width, button_height, highlight_text_color, highlight_background_color, caption, normalTextSize);
}

void tftDisplayMediumLengthRightButton(int yNdx, String caption) {
  tftDisplayButton(load_button_x0 + di_buttons_width + di_buttons_space, button_y0 + yNdx * button_height, di_buttons_width, button_height, highlight_text_color, highlight_background_color, caption, normalTextSize);
}

void tftDisplayMediumLengthLeftButton(int yNdx, String caption, uint16_t bkColor, uint16_t fgColor) {
  tftDisplayButton(load_button_x0, button_y0 + yNdx * button_height, di_buttons_width, button_height, fgColor, bkColor, caption, normalTextSize);
}

void tftDisplayMediumLengthRightButton(int yNdx, String caption, uint16_t bkColor, uint16_t fgColor) {
  tftDisplayButton(load_button_x0 + di_buttons_width + di_buttons_space, button_y0 + yNdx * button_height, di_buttons_width, button_height, fgColor, bkColor, caption, normalTextSize);
}

void tftDisplayInterfaceDisconnectButtons(int yNdx) {
  if (!connectToggleButtonPressed)
    return;
  connectToggleButtonPressed = false;
  tftEraseDiButtons(yNdx);
  String s;
  uint16_t textColor;
  if (txInterfaceDisconnect) {
    s = " Connect Tx";
    textColor = TFT_ALT;
  } else {
    s = " Unplug Tx";
    textColor = TFT_TXT;
  }
  tftDisplayMediumLengthLeftButton(yNdx, s, uColor(LT_GREY), textColor);
  if (rxInterfaceDisconnect) {
    s = " Connect Rx";
    textColor = TFT_ALT;
  } else {
    s = " Unplug Rx";
    textColor = TFT_TXT;
  }
  tftDisplayMediumLengthRightButton(yNdx, s, uColor(LT_GREY), textColor);
}

void tftDisplayMultiwayInterfaceButtons(int yNdx) {
  uint16_t lockTextColor = BLUE;
  uint16_t connectTextColor = RED;
  uint16_t unplugTextColor = uColor(BROWN);
  uint16_t bkg = uColor(LT_GREY);
  uint16_t textColor;
  String s;
  // Left Button
  if (txFrequencyLock) {
    if (txInterfaceDisconnect) {
      // true / true (unreachable)
      // Show Connect Tx button
      s = " Connect Tx";
      textColor = connectTextColor;
    } else {
      // true / false -> false / true
      // Show Unplug Tx button
      s = " Unplug Tx";
      textColor = unplugTextColor;
    }
  } else {
    if (txInterfaceDisconnect) {
      // false / True -> false / false
      // Show Connect Tx button
      s = " Connect Tx";
      textColor = connectTextColor;
    } else {
      // false / false -> true / false
      // Show Lock Tx button
      s = "  Lock Tx";
      textColor = lockTextColor;
    }
  }
  tftDisplayMediumLengthLeftButton(yNdx, s, bkg, textColor);
  // Right Button
  if (rxFrequencyLock) {
    if (rxInterfaceDisconnect) {
      // true / true (unreachable)
      // Show Connect Rx button
      s = " Connect Rx";
      textColor = connectTextColor;
    } else {
      // true / false -> false / true
      // Show Unplug Rx button
      s = " Unplug Rx";
      textColor = unplugTextColor;
    }
  } else {
    if (rxInterfaceDisconnect) {
      // false / True -> false / false
      // Show Connect Rx button
      s = " Connect Rx";
      textColor = connectTextColor;
    } else {
      // false / false -> true / false
      // Show Lock Rx button
      s = "  Lock Rx";
      textColor = lockTextColor;
    }
  }
  tftDisplayMediumLengthRightButton(yNdx, s, bkg, textColor);
}

void tftDisplayCancelSaveButtons(int yNdx) {
  tftDisplayMediumLengthLeftButton(yNdx, "   Cancel");
  tftDisplayMediumLengthRightButton(yNdx, "    Save");
}

void tftDisplayGenericFullLengthButton(String caption) {
  tft.fillRoundRect(load_button_x0, button_y0 + (MAX_SATS + 1) * button_height, load_button_width, button_height, button_radius, highlight_background_color);
  tftDisplayButton(load_button_x0, button_y0 + (MAX_SATS + 1) * button_height, load_button_width, button_height, highlight_text_color, highlight_background_color, caption, normalTextSize);
}

void tftDisplayGenericHexaButton(int btnNumber, int yNdx, String caption) {
  tftDisplayButton(load_button_x0 + (btnNumber) * (hexa_buttons_width + hexa_buttons_space), button_y0 + yNdx * button_height, hexa_buttons_width, button_height, highlight_text_color, highlight_background_color, caption, normalTextSize);
}

void tftDisplayGenericHexaButton(int btnNumber, int yNdx, String caption, uint16_t bkColor, uint16_t fgColor) {
  tftDisplayButton(load_button_x0 + (btnNumber) * (hexa_buttons_width + hexa_buttons_space), button_y0 + yNdx * button_height, hexa_buttons_width, button_height, fgColor, bkColor, caption, normalTextSize);
}

void tftDisplayTestButtons(int yNdx) {
  // Test buttons are conditionally displayed in the tools screen
  // if and only if debug DIP switch is set on power-on
  String s = "T0";
  uint16_t bkgColor;
  for (int i = 0; i < MAX_TEST_CONDITIONS; i++) {
    if (test_condition[i] == true)
      bkgColor = GREEN;
    else
      bkgColor = RED;
    tftDisplayGenericHexaButton(i, yNdx, s + String(i), bkgColor, WHITE);
  }
}

void tftEraseLoadSatelliteButton() {
  tft.fillRect(load_button_x0, button_y0 + (MAX_SATS + 1) * button_height, load_button_width, button_height, mainscreen_background_color);
}

void tftEraseSatelliteDataButton(int button_index) {
  // Screen 2 button specified by vertical position
  tft.fillRect(load_button_x0, button_y0 + button_index * button_height, load_button_width, button_height, satData_background_color);
}

void tftDisplaySelectedSatelliteData(int yPos, String sData, uint16_t sz) {
  // Erase previous
  //tft.fillRoundRect(sat_data_x0, yPos, X_PIXELS-sat_data_x0, button_half_height*sz, button_radius, satData_background_color);
  tft.fillRect(0, yPos, X_PIXELS, button_half_height * sz, satData_background_color);
  // Display textual data
  tft.setTextColor(satData_text_color);
  tft.setTextSize(sz);
  tft.setCursor(sat_data_x0, yPos);
  tft.print(sData);
}

void tftDisplaySelectedSatelliteData(int yNdx, String sData) {
  tftDisplaySelectedSatelliteData(yNdx * button_height, sData, 2);
}

void tftEraseSatelliteDataLine(int line_index) {
  // Screen 2 text line specified by vertical position
  tft.fillRect(0, line_index * button_height, X_PIXELS, button_height, satData_background_color);
}

void tftDisplaySelectedSatellite() {
  tftDisplayButton(button_x0, button_y0, button_width, button_height, highlight_text_color, highlight_background_color, satNames[selectedSatellite], normalTextSize);
}

void tftDisplayReturnToSatellitesButton() {
  tftDisplayButton(load_button_x0, button_y0 + (MAX_SATS + 1) * button_height, load_button_width, button_height, highlight_text_color, highlight_background_color, "  Return to satellites", normalTextSize);
}

void tftDisplayTxDopplerSendToggle() {
  uint16_t text_color = RED;
  uint16_t background_color = uColor(LT_GREY);
  if (txDopplerCorrectionEnabled)
    text_color = GREEN;
  tftDisplayButton(load_button_x0, button_y0 + txDopplerSwitchYindex * button_height, load_button_width, button_height, text_color, background_color, "   Tx Doppler On/Off", normalTextSize);
}

void tftDisplayRxDopplerSendToggle() {
  uint16_t text_color = RED;
  uint16_t background_color = uColor(LT_GREY);
  if (rxDopplerCorrectionEnabled)
    text_color = GREEN;
  tftDisplayButton(load_button_x0, button_y0 + rxDopplerSwitchYindex * button_height, load_button_width, button_height, text_color, background_color, "   Rx Doppler On/Off", normalTextSize);
}

void tftEraseFullLengthButton(int yNdx, uint16_t bkg) {
  tft.fillRoundRect(load_button_x0, button_y0 + yNdx * button_height, load_button_width, button_height, button_radius, bkg);
}

void tftDisplayFrequencyCalibrationButtons() {
  uint16_t bkg = vfoCalibrationButtonsBackgroundColor;
  uint16_t txt = vfoCalibrationButtonsTextColor;
  if (txCalibrateEnabled)
    tftDisplayGenericHexaButton(0, vfoCalibrationYindex, "Tx", bkg, txt);
  else if (rxCalibrateEnabled)
    tftDisplayGenericHexaButton(0, vfoCalibrationYindex, "Rx", bkg, txt);
  tftDisplayGenericHexaButton(vfoCalibrationSubButtonXindex, vfoCalibrationYindex, "Sub", bkg, txt);
  tftDisplayGenericHexaButton(vfoCalibrationAddButtonXindex, vfoCalibrationYindex, "Add", bkg, txt);
}

void tftDisplayFrequencyCalibrationValue() {
  String s = "";
  // Next declarations are temporary - To be parameterized
  uint16_t rectX = load_button_x0 + 3 * (hexa_buttons_width + hexa_buttons_space);
  uint16_t cValX = rectX + hexa_buttons_space;
  uint16_t yPos = button_y0 + vfoCalibrationYindex * button_height;
  uint16_t units = cValX + 60;
  if (txCalibrateEnabled)
    s.concat(String(txIntercept));
  else if (rxCalibrateEnabled)
    s.concat(String(rxIntercept));
  // Following specifications to be parameterized
  tft.fillRect(rectX, yPos, 2 * hexa_buttons_width + hexa_buttons_space, button_height, toolscreen_background_color);
  tft.drawRect(rectX, yPos, 2 * hexa_buttons_width + hexa_buttons_space, button_height, WHITE);
  tft.setCursor(cValX, yPos + 2);
  tft.setTextColor(WHITE);
  tft.setTextSize(2);
  tft.print(s);
  tft.setCursor(units, yPos + 2);
  tft.print("Hz");
}

void tftEraseFrequencyCalibrationComponents() {
  tftEraseFullLengthButton(vfoCalibrationYindex, toolscreen_background_color);
}

void tftDisplayFrequencyCalibrationComponents() {
  tftDisplayFrequencyCalibrationButtons();
  tftDisplayFrequencyCalibrationValue();
}

// ----------------------------------TFT-Touch-------------------------------------

boolean tftAbort() {                          // Exit screensaver here
  // Minimize time here
  // return ts.touched();
  if (ts.touched()) {
    while (ts.touched())
      delay(MOMENT);
    lastTouched = millis();
    delay(DEBOUNCE_TOUCH);                    // Additional debounce
    if (screenSaver) {
//    Serial.println("In tftAbort() - screenSaver true");
      screenSaver = false;
      if (screenID == 1)
        loadMainScreen();
      else if (screenID == 3)
        loadToolsScreen();
      return true;
    }
    else 
      return false;
  }
  else
    return false;
}

void tftProcessTouch() {
  while (ts.touched())
    ;  // Debounce
  lastTouched = millis();
  delay(DEBOUNCE_TOUCH);                      // Additional debounce
  if (screenSaver) {                          
//  Serial.println("In tftProcessTouch() - screenSaver true");
    screenSaver = false;
    if (screenID == 1)
      loadMainScreen();
    else if (screenID == 3)
      loadToolsScreen();
    return;
  }
  touch_being_processed = true;
  ts.readData(&ts_x, &ts_y, &ts_z);
  tftCompleteTouchProcessing();
  touch_being_processed = false;
}

void tftCompleteTouchProcessing() {
  // Handle screen saver if implemented
  int itemp;
  int yNdx = (ts_y - 400) / 300;
  if (screenID == 0) {
    screenID = 1;
    tftDisplaySatelliteNames();
  } else if (screenID == 1) {  // Main screen
    itemp = tftSelectedSatellite();
    if (itemp < satCount) {
      if (selectedSatellite == itemp) {  // If already selected, toggle OFF
        tftDisplayButton(button_x0, button_y0 + itemp * button_height, button_width, button_height, MAGENTA, uColor(LT_GREY), satNames[itemp], normalTextSize);
        selectedSatellite = -1;
        tftEraseLoadSatelliteButton();
        tftDisplayTriButtons();
      } else {
        selectedSatellite = itemp;
        tftHighlightSatelliteName(selectedSatellite);
        tftEraseTriButtons();
        tftDisplayLoadSatelliteButton();
      }
    } else if (itemp == loadButtonIndex) {
      if (selectedSatellite >= 0) {
        loadSelectedSatellite();
        processSelectedSatellite();
      } else {
        // Fall through here when no satellite name is highlighted
        if (ts_x < 1400) {
          loadToolsScreen();
          //        tftDisplayStatusbar("Tools touched"); // Placeholders
        } else if (ts_x < 2500) {
          tftDisplayStatusbar("More touched");
        } else {
          tftDisplayStatusbar("Back touched");
        }
      }
    }
  } else if (screenID == 2) {  // Selected satellite screen
    // Detect return to satellites button touch
    if (yNdx == 9) {
      pass_in_progress = false;  // Temporary kludge
      lastSelectedSatellite = selectedSatellite;
      loadMainScreen();
    } else if (yNdx == txDopplerSwitchYindex && pass_in_progress) {
      txDopplerCorrectionEnabled = !txDopplerCorrectionEnabled;
      sendTransmitFrequencyUpdate();  // Update immediately - Do not wait for frequency change
      tftDisplayTxDopplerSendToggle();
    } else if (yNdx == rxDopplerSwitchYindex && pass_in_progress) {
      rxDopplerCorrectionEnabled = !rxDopplerCorrectionEnabled;
      sendReceiveFrequencyUpdate();  // See note above
      tftDisplayRxDopplerSendToggle();
    } else if (yNdx == lockUnplugSwitchesYindex && pass_in_progress) {
      // toggleInterfaceButtonTouched();
      multiwayInterfaceButtonTouched();
    }
  } else if (screenID == 3) {
    // Detect return to satellites button touch
    if (yNdx == returnButtonYindex) {
      loadMainScreen();  // Context is set in called function
    } else if (yNdx == vfoCalibrationYindex) {
      processVFOcalibrationTool();
    } else if (displayTestButtons && yNdx == testButtonsYindex) {
      processTestButtons();
    } else {
      adjustEditableTime();
      adjustRTCtime();
    }
  }
}

int tftSelectedSatellite() {
  return (ts_y - 400) / 300;
}

int tftTimeAdjustButtonTouched() {
  int buttonID = 0;
  if (ts_y < 1300) {
    if (ts_y < 800)
      buttonID = 10;
    else
      buttonID = 20;
    if (ts_x < 850)
      buttonID += 1;
    else if (ts_x < 1420)
      buttonID += 2;
    else if (ts_x < 1960)
      buttonID += 3;
    else if (ts_x < 2500)
      buttonID += 4;
    else if (ts_x < 3050)
      buttonID += 5;
    else
      buttonID += 6;
  }
  return buttonID;
}

int tftHexaButtonTouched() {
  if (ts_x < 850)
    return 0;
  else if (ts_x < 1420)
    return 1;
  else if (ts_x < 1960)
    return 2;
  else if (ts_x < 2500)
    return 3;
  else if (ts_x < 3050)
    return 4;
  else
    return 5;
}

int tftTestButtonTouched() {
  return tftHexaButtonTouched();
}

int timeAdjustCancelSaveButtonTouched(int yNdx) {
  // 1 = cancel
  // 2 = save
  int top = yNdx * 325 + 300;
  if (ts_y > top && ts_y < top + 325) {
    if (ts_x < 1950)
      return 1;
    else
      return 2;
  } else return 0;
}

void toggleInterfaceButtonTouched() {
  if (ts_x < 1925) {
    txInterfaceDisconnect = !txInterfaceDisconnect;
  } else {
    rxInterfaceDisconnect = !rxInterfaceDisconnect;
  }
  connectToggleButtonPressed = true;
}

void multiwayInterfaceButtonTouched() {
  if (ts_x < 1925) {
    // Transmit multiway button touched
    if (txFrequencyLock) {
      if (txInterfaceDisconnect) {
        // true / true (unreachable)
      } else {
        // true / false -> false / true
        txFrequencyLock = false;
        txInterfaceDisconnect = true;
        resyncTransmitVFO();
      }
    } else {
      if (txInterfaceDisconnect) {
        // false / True -> false / false
        txFrequencyLock = false;
        txInterfaceDisconnect = false;
      } else {
        // false / false -> true / false
        txFrequencyLock = true;
        txInterfaceDisconnect = false;
      }
    }
  } else {
    // Receive multiway button touched
    if (rxFrequencyLock) {
      if (rxInterfaceDisconnect) {
        // true / true (unreachable)
      } else {
        // true / false -> false / true
        rxFrequencyLock = false;
        rxInterfaceDisconnect = true;
        resyncReceiveVFO();
      }
    } else {
      if (rxInterfaceDisconnect) {
        // false / True -> false / false
        rxFrequencyLock = false;
        rxInterfaceDisconnect = false;
      } else {
        // false / false -> true / false
        rxFrequencyLock = true;
        rxInterfaceDisconnect = false;
      }
    }
  }
  connectToggleButtonPressed = true;
  tftDisplayMultiwayInterfaceButtons(lockUnplugSwitchesYindex);  // To do: Encapsulate
}

String sTouchData() {
  String s = "";
  s.concat(String(ts_x));
  s.concat(", ");
  s.concat(String(ts_y));
  s.concat(", ");
  s.concat(String(ts_z));
  return s;
}

String statusMessageText() {
  if ((0 <= statusMessageNumber) && (statusMessageNumber < MAX_CANNED_STATUS_MESSAGES))
    return statusMessage[statusMessageNumber];  // Placeholder
  else                                          // Placeholder
    return sTouchData();                        // Default status message
}

// -------------------------------------ISR----------------------------------------

void isrA() {
  // This ISR reached when ROTARY_ENCODER_A transitions from LOW to HIGH
  // Check ROTARY_ENCODER_B to determine the direction
  if (digitalRead(ROTARY_ENCODER_B) == LOW)
    rawCount++;
  else
    rawCount--;
}

void isrB() {
  // This ISR reached when ROTARY_ENCODER_B transitions from LOW to HIGH
  // Check ROTARY_ENCODER_A to determine the direction
  if (digitalRead(ROTARY_ENCODER_A) == LOW)
    rawCount--;
  else
    rawCount++;
}

// ---------------------------------Interfaces-------------------------------------

void initRadioInterfaces() {
  rigInterface0.begin(rig0BAUD);
  rigInterface1.begin(rig1BAUD);
}

void nullModemTest(int rig_number) {
  if (rig_number == 0)
    for (unsigned int i = 0; i < loopbackTest.length(); i++)
      rigInterface0.write(loopbackTest.charAt(i));
  else if (rig_number == 1)
    for (unsigned int i = 0; i < loopbackTest.length(); i++)
      rigInterface1.write(loopbackTest.charAt(i));
}
void nullModemTest() {
  // Default to testing both rig interfaces
  // No harm, if either is not connected
  nullModemTest(0);
  nullModemTest(1);
}

void zeroRxBuffer() {
  serRxBuffer[0] = (byte)0;
  rxNdx = 0;
  rigCommandReceived = false;
}

void rigInterface0Listen() {  // Not used - Currently only CI-V - No processing
  if (txInterfaceDisconnect) {
    return;
  }
  if (rxNdx >= RX_BUFFER_DIM)
    zeroRxBuffer();
  byte data_byte;
  while (rigInterface0.available()) {
    data_byte = rigInterface0.read();
    serRxBuffer[rxNdx++] = data_byte;
    if (data_byte == EOM) {
      // To do: Verify sender is not self (3rd byte in message)
      rigCommandReceived = true;
      return;
    }
  }
}

void rigInterface1Listen() {  // Debug only - Currently only CAT - No processing
  if (rxInterfaceDisconnect) {
    return;
  }
  if (rxNdx >= RX_BUFFER_DIM)
    zeroRxBuffer();
  byte data_byte;
  while (rigInterface1.available()) {
    data_byte = rigInterface1.read();
    if (data_byte > 31) {
      serRxBuffer[rxNdx++] = data_byte;
      if (data_byte == EOC) {
        rigCommandReceived = true;
        return;
      }
    }
  }
}

void processReceivedCivMessage() {
  // E.g. CI-V OK message
  // FE FE E0 88 FB FD
  //             ^____ Fixed OK code
  //
  for (int i = 0; i < rxNdx; i++) {
    Serial.print(serRxBuffer[i], HEX);
    Serial.print(SP);
  }
  Serial.println();
}

void sendReceiveFrequencyUpdate() {
  if (rxInterfaceDisconnect) {
    return;
  }
  //Serial.println("Receive Interface Enabled");
  String s = "FA";
  String t;
  if (rxDopplerCorrectionEnabled)
    t = String(receiveFrequencyDopplerVFO);
  else
    t = String(receiveFrequencyVFO);
  if (rcvInterface) {
    if (rcvType == "CAT") {  // Currently only FT-991A CAT interface supported
      while (t.length() < 9)
        s.concat('0');
      if (t.length() > 9)
        return;  // FT991-A CAT frequency must be exactly 9 digits
      s.concat(t);
      s.concat((char)EOC);
      for (unsigned int i = 0; i < s.length(); i++)
        rigInterface1.write(s.charAt(i));
    }
    // To do: CI-V and other rig types
  }
}

void sendTransmitFrequencyUpdate() {  // Currently only Icom CI-V interface supported
  if (txInterfaceDisconnect) {
    return;
  }
  //Serial.println("Transmit Interface Enabled");
  String s;
  if (xmtInterface) {
    if (xmtType == "CAT")
      return;
  } else
    return;
  // Confirmed CI-V interface
  if (txDopplerCorrectionEnabled)
    cat2civFreqFormat(String(transmitFrequencyDopplerVFO));
  else
    cat2civFreqFormat(String(transmitFrequencyVFO));
  if (CIV_Frequency.length() == 5) {
    if (CIV_Frequency == lastCIV_Frequency)
      return;
    lastCIV_Frequency = CIV_Frequency;
    /*
  Serial.print("In sendTransmitFrequencyUpdate() - CIV_Frequency after conversion: ");
  for (int i=0; i<5; i++) {
    Serial.print((byte) CIV_Frequency.charAt(i), HEX);
    Serial.print(" ");
  }
  Serial.println();
*/
    s = civFreqCommand();
    Serial.print("Send to Tx: ");
    for (unsigned int i = 0; i < s.length(); i++) {
      rigInterface0.write(s.charAt(i));
      Serial.print(s.charAt(i), HEX);
      Serial.print(' ');
    }
    rigInterface0.write(CR);
    rigInterface0.write(LF);
    Serial.println();
  } else {
    Serial.print("Invalid CIV Frequency: ");
    Serial.println(CIV_Frequency);
  }
  return;
}

void sendAntennaAzimuth() {  // Send rotor command Mxxx
  // String satAz was declared globally and updated by the caller
  String s = satAz.substring(0, satAz.indexOf(POINT));
  if (s.length() < 1)
    return;
  if (rotorInterfaceType == 0) {
    while (s.length() < 3)
      s = ZERO + s;
    antInterface.write('M');
    for (unsigned int i = 0; i < 3; i++)
      antInterface.write(s.charAt(i));
    antInterface.write(CR);
  } else if (rotorInterfaceType == 1) {
    antInterface.write('A');
    antInterface.write('Z');
    for (unsigned int i = 0; i < s.length(); i++)
      antInterface.write(s.charAt(i));
    antInterface.write(CR);
  } else
    return;
}

void cat2civFreqFormat(String catFreq) {
  /*
  Serial.print("In cat2civFreqFormat(String catFreq) - Raw catFreq: ");
  Serial.println(catFreq);
*/
  if (catFreq.length() > 9)
    return;  // Unexpected - Do nothing
  while (catFreq.length() < 10)
    catFreq = "0" + catFreq;  // Pad up to 1000's of MHz
                              /*
  Serial.print("catFreq to be converted: ");
  Serial.println(catFreq);
*/
  CIV_Frequency = "";
  byte b, left, right;
  for (unsigned int i = 10; i > 0; i -= 2) {
    right = (byte)catFreq.charAt(i - 1) - 48;
    left = (byte)catFreq.charAt(i - 2) - 48;
    b = left * 16 + right;  // to BCD
    CIV_Frequency.concat((char)b);
  }
}

String civFreqCommand(String civFreq) {
  // Wrap CI-V format frequency to construct full Icom CI-V frequency command.
  String civMsg = "";
  civMsg.concat((char)0xFE);
  civMsg.concat((char)0xFE);
  civMsg.concat((char)civAddress());
  civMsg.concat((char)0xE0);  // Controller default address
  civMsg.concat((char)0x00);  // Frequency command group
  // Pre-formatted CI-V frequency
  for (unsigned int i = 0; i < civFreq.length(); i++)
    civMsg.concat(civFreq.charAt(i));
  civMsg.concat((char)EOM);
  return civMsg;
}

String civFreqCommand() {
  // Wrap CI-V format frequency to construct full Icom CI-V frequency command.
  // Uses global String CIV_Frequency
  String civMsg = "";
  civMsg.concat((char)0xFE);
  civMsg.concat((char)0xFE);
  civMsg.concat((char)civAddress());
  civMsg.concat((char)0xE0);  // Controller default address
  civMsg.concat((char)0x00);  // Frequency command group
  // Pre-formatted CI-V frequency
  for (unsigned int i = 0; i < CIV_Frequency.length(); i++)
    civMsg.concat(CIV_Frequency.charAt(i));
  civMsg.concat((char)EOM);
  return civMsg;
}

byte civAddress() {
  if (xmtType != "CAT") {
    if (xmtType == "88h")  // IC-7100
      return 0x88;
    else if (xmtType == "94h")  // IC-7300
      return 0x94;
    else if (xmtType == "98h")  // IC-7610
      return 0x98;
    else if (xmtType == "A2h")  // IC-9700
      return 0xA2;
    else
      return 0;
  } else
    return 0;
}


// ----------------------------------File I/O--------------------------------------

void loadFileList() {
  if (SD_card_enabled) {
    File root = SD.open("/");
    File entry;
    String fileName;
    for (fileCount = 0; fileCount < MAX_FILES; fileCount++) {
      entry = root.openNextFile();
      if (!entry)
        break;
      // Skip directories
      if (entry.isDirectory()) {
        fileCount--;
      } else {
        fileName = entry.name();
        for (unsigned int i = 0; i < fileName.length(); i++) {
          fileList[fileCount].concat(fileName.charAt(i));
        }
      }
      entry.close();
    }
    root.close();
  }
}

boolean isTLEfile() {
  return fileExists("amateur.txt");
}

boolean fileExists(String filename) {
  for (unsigned int i = 0; i < fileCount; i++) {
    if (fileList[i].length() == 0)
      return false;
    else if (fileList[i] == filename)
      return true;
  }
  return false;
}

void initTextFileData() {
  for (int i = 0; i < MAX_LINES; i++)
    textFileData[i] = "";
}

void initFreqList() {
  for (int i = 0; i < MAX_SATS; i++)
    freqList[i] = "";
}

void initStationParams() {
  for (int i = 0; i < MAX_PARAMS; i++)
    stationParams[i] = "";
}

void readFrequencies() {
  initFreqList();
  File frequencies = SD.open("frequencies.txt");
  byte b;
  if (frequencies) {
    while (frequencies.available()) {
      b = frequencies.read();
      if (b >= 32 || (b == TAB))
        freqList[satCount].concat((char)b);
      else if (b == EOL) {
        satCount++;
      }
    }
  }
  frequencies.close();
}

void readTLEs() {
  initTextFileData();
  File TLEs = SD.open("amateur.txt");
  byte b;
  int i = 0;
  if (TLEs) {
    while (TLEs.available()) {
      b = TLEs.read();
      if (b >= 32 || b == TAB)
        TLEdata[i].concat((char)b);
      else if (b == EOL) {
        i++;
      }
    }
  }
  TLEs.close();
}

void indexTLEs() {
  String satName;
  String tleRecord;
  for (int i = 0; i < MAX_SATS; i++) {
    satName = satNames[i];
    if (satName == "")
      return;
    for (int j = 0; j < MAX_TLE * 3; j++) {
      tleRecord = TLEdata[j];
      if (tleRecord.indexOf(satName) < 0)
        continue;
      else {
        satTLEindex[i] = j;
        break;
      }
    }
  }
}

void loadSelectedTLE(int iSat) {
  // Sat is the index number of the selected satellite
  Serial.println();
  Serial.print("loadSelectedTLE(");
  Serial.print(iSat);
  Serial.println("):");
  int ndx = satTLEindex[iSat];
  Serial.print("ndx: ");
  Serial.println(ndx);
  Serial.println(TLEdata[ndx]);
  currentTLE[0] = TLEdata[++ndx];
  currentTLE[1] = TLEdata[++ndx];
}

void initSatNames() {
  String satName;
  for (int i = 0; i < satCount; i++) {
    if (freqList[i] == "")
      return;
    satName = freqList[i].substring(0, freqList[i].indexOf(TAB));
    satNames[i] = satName;
  }
}

void readStationParams() {
  initStationParams();
  File params = SD.open("station.txt");
  byte b;
  int i = 0;
  if (params) {
    while (params.available()) {
      b = params.read();
      if (b >= 32 || b == TAB)
        stationParams[i].concat((char)b);
      else if (b == EOL) {
        i++;
      }
    }
  }
  params.close();
}

void extractStationParameters() {
  String s;
  sCallsign = stationParams[0].substring(0, stationParams[0].indexOf(TAB));
  s = stationParams[1].substring(0, stationParams[1].indexOf(TAB));
  if (s.charAt(0) == MINUS)
    fLat = -s.substring(1).toFloat();
  else
    fLat = s.toFloat();
  s = stationParams[2].substring(0, stationParams[2].indexOf(TAB));
  if (s.charAt(0) == MINUS)
    fLon = -s.substring(1).toFloat();
  else
    fLon = s.toFloat();
  s = stationParams[3].substring(0, stationParams[3].indexOf(TAB));
  if (s.charAt(0) == MINUS)
    fAlt = -s.substring(1).toFloat();
  else
    fAlt = s.toFloat();
  s = stationParams[4].substring(0, stationParams[4].indexOf(TAB));
  if (s.charAt(0) == MINUS)
    fTZ = -s.substring(1).toFloat();
  else
    fTZ = s.toFloat();
  s = stationParams[5].substring(0, stationParams[5].indexOf(TAB));
  DST_hours = s.toInt();
  if (DST_hours == 0)
    DST = false;
  else
    DST = true;
  s = stationParams[6].substring(0, stationParams[6].indexOf(TAB));
  if (s.charAt(0) == '1')
    xmtInterface = true;
  else
    xmtInterface = false;
  s = stationParams[7].substring(0, stationParams[7].indexOf(TAB));
  if (s.charAt(0) == '1')
    rcvInterface = true;
  else
    rcvInterface = false;
  xmtType = stationParams[8].substring(0, stationParams[8].indexOf(TAB));
  rcvType = stationParams[9].substring(0, stationParams[9].indexOf(TAB));
  s = stationParams[10].substring(0, stationParams[10].indexOf(TAB));
  xmtBAUD = (uint32_t)s.toInt();
  s = stationParams[11].substring(0, stationParams[11].indexOf(TAB));
  rcvBAUD = (uint32_t)s.toInt();
  tuningPreference = stationParams[12].substring(0, stationParams[12].indexOf(TAB));
  DopplerPreference = stationParams[13].substring(0, stationParams[13].indexOf(TAB));
}

void readMicroSD() {
  readFrequencies();
  readTLEs();
  readStationParams();
  extractStationParameters();
}

void initConvenienceArrays() {
  initSatNames();
  indexTLEs();
}


// ------------------------------Load/Process Screens------------------------------

void loadMainScreen() {
  // First screen after splash()
  // Or return from processing a selected satellite
  tft.fillScreen(mainscreen_background_color);
  tftDisplaySatelliteNames();
  tftDisplayTriButtons();
  selectedSatellite = -1;       // Reset on return from selected satellite context
  if (context_free_send_to_Tx)  // Ditto
    txInterfaceDisconnect = false;
  if (context_free_send_to_Rx)
    rxInterfaceDisconnect = false;
  screenID = 1;
}

void loadToolsScreen() {
  // Selected from main screen
  // Returns to main screen
  tft.fillScreen(toolscreen_background_color);
  zeroDeltaDateTime();
  for (int i = 0; i < 6; i++) {  // i = zero-based button number (left to right)
    tftDisplayGenericHexaButton(i, 0, " Up");
  }
  for (int i = 0; i < 6; i++) {
    tftDisplayGenericHexaButton(i, 2, "Dwn");
  }
  if (displayTestButtons)
    tftDisplayTestButtons(testButtonsYindex);
  if (calibrationToolEnabled)
    tftDisplayFrequencyCalibrationComponents();
  tftDisplayGenericFullLengthButton(" Return to Satellites");
  screenID = 3;
}

void tftDisplayTriButtons() {
  tftDisplayGenericLeftButton(" Tools");
  tftDisplayGenericMiddleButton("  More");
  tftDisplayGenericRightButton("  Back");
}

void tftEraseTriButtons() {
  tft.fillRect(load_button_x0, button_y0 + (MAX_SATS + 1) * button_height, 3 * tri_buttons_width + 2 * tri_buttons_space, button_height, mainscreen_background_color);
}

void tftEraseDiButtons(int yNdx) {
  tft.fillRect(load_button_x0, button_y0 + yNdx * button_height, 2 * di_buttons_width + di_buttons_space, button_height, mainscreen_background_color);
}

void loadSelectedSatellite() {
  loadSelectedTLE(selectedSatellite);
  selectedSatelliteFrequencies = freqList[selectedSatellite];
  if (lastSelectedSatellite != selectedSatellite)
    setSelectedNumericFrequencies();
}

void setSelectedNumericFrequencies() {
  // 'transmit' => uplink, 'receive' => downlink
  String s = selectedSatelliteFrequencies.substring(selectedSatelliteFrequencies.indexOf(TAB) + 1);
  int n;
  if (s == "")
    return;
  transmitFrequencyLB = s.substring(0, s.indexOf(TAB)).toInt();
  transmitFrequencyLB *= 1000;
  n = s.indexOf(TAB);
  if (n >= 0)
    s = s.substring(s.indexOf(TAB) + 1);
  else
    s = "";
  transmitFrequencyUB = s.substring(0, s.indexOf(TAB)).toInt();
  transmitFrequencyUB *= 1000;
  n = s.indexOf(TAB);
  if (n >= 0)
    s = s.substring(s.indexOf(TAB) + 1);
  else
    s = "";
  receiveFrequencyLB = s.substring(0, s.indexOf(TAB)).toInt();
  receiveFrequencyLB *= 1000;
  if (transmitFrequencyLB < transmitFrequencyUB)
    transmitFrequencyInverted = false;
  else
    transmitFrequencyInverted = true;
  if (receiveFrequencyLB < receiveFrequencyUB)
    receiveFrequencyInverted = false;
  else
    receiveFrequencyInverted = true;

  n = s.indexOf(TAB);
  if (n >= 0)
    s = s.substring(s.indexOf(TAB) + 1);
  else
    s = "";
  receiveFrequencyUB = s.substring(0, s.indexOf(TAB)).toInt();
  receiveFrequencyUB *= 1000;
  n = s.indexOf(TAB);
  if (n >= 0)
    s = s.substring(s.indexOf(TAB) + 1);
  else
    s = "";
  transmitFrequencyVFO = s.substring(0, s.indexOf(TAB)).toInt();
  transmitFrequencyVFO *= 1000;
  if (transmitFrequencyVFO == 0)
    transmitFrequencyVFO = (transmitFrequencyLB + transmitFrequencyUB) / 2;
  n = s.indexOf(TAB);
  if (n >= 0)
    s = s.substring(s.indexOf(TAB) + 1);
  else
    s = "";
  receiveFrequencyVFO = s.substring(0, s.indexOf(TAB)).toInt();
  receiveFrequencyVFO *= 1000;
  if (receiveFrequencyVFO == 0)
    receiveFrequencyVFO = (receiveFrequencyLB + receiveFrequencyUB) / 2;
  // For context-free updating ->
  lastReceiveFrequencyVFO = receiveFrequencyVFO;
}

void processVFOcalibrationTool() {
  int buttonID = tftHexaButtonTouched();
  if (buttonID == 0) {
    if (txCalibrateEnabled) {
      txCalibrateEnabled = false;
      rxCalibrateEnabled = true;
    } else if (rxCalibrateEnabled) {
      rxCalibrateEnabled = false;
      txCalibrateEnabled = true;
    }
    tftDisplayFrequencyCalibrationButtons();
    tftDisplayFrequencyCalibrationValue();
  } else if (buttonID == vfoCalibrationSubButtonXindex) {
    if (txCalibrateEnabled)
      txIntercept -= vfoCalibrationDelta;
    else if (rxCalibrateEnabled)
      rxIntercept -= vfoCalibrationDelta;
    tftDisplayFrequencyCalibrationValue();
  } else if (buttonID == vfoCalibrationAddButtonXindex) {
    if (txCalibrateEnabled)
      txIntercept += vfoCalibrationDelta;
    else if (rxCalibrateEnabled)
      rxIntercept += vfoCalibrationDelta;
    tftDisplayFrequencyCalibrationValue();
  }
  eeSaveIntercepts();
}

void processSelectedSatellite() {
  // Transition to screen 2 - Satellite details, etc.
  tft.fillScreen(satData_background_color);
  tftDisplaySelectedSatellite();
  tftDisplayReturnToSatellitesButton();
  initInterfaceSwitches();
  displayVFO();
  initSGP4();
  Serial.println();
  Serial.println("Satellite: " + satNames[selectedSatellite]);
  secondsCounter = 0;
  satLastRangeTimestamp = 0;          // Reset for each selected satellite
  connectToggleButtonPressed = true;  // Repaint on entry
  screenID = 2;
}

void initInterfaceSwitches() {
  rxFrequencyLock = false;
  txFrequencyLock = false;
  rxInterfaceDisconnect = false;
  txInterfaceDisconnect = false;
}

// ------------------------------------SGP4----------------------------------------

void initSGP4() {
  // Require station latitude/longitude and selected satellite TLE's
  const int MAX_DIM = 80;
  char satID[MAX_DIM], TLE1[MAX_DIM], TLE2[MAX_DIM];

  if (selectedSatellite < 0)
    return;

  for (unsigned int i = 0; i < satNames[selectedSatellite].length(); i++)
    satID[i] = satNames[selectedSatellite].charAt(i);
  for (unsigned int i = 0; i < currentTLE[0].length(); i++)
    TLE1[i] = currentTLE[0].charAt(i);
  for (unsigned int i = 0; i < currentTLE[1].length(); i++)
    TLE2[i] = currentTLE[1].charAt(i);

  sat.site(fLat, fLon, fAlt);
  sat.init(satID, TLE1, TLE2);
}

void updateSatelliteInfo() {
  float rxD, txD;  // function return (Doppler adjusted frequencies)
  float rxDisplayDoppler, txDisplayDoppler;
  String sDataLine = "";
  boolean ascending = false;
  boolean descending = false;
  double jd_of_predicted_pass = 0.;
  unsigned long ut_of_predicted_pass = 0;
  int checkDST;
  int verifiedDSThours = DST_hours;
  int thisSecond = second();

  if (thisSecond == lastSgp4Tick)  // Sync updating to Teensy RTC
    return;
  lastSgp4Tick = thisSecond;
  secondsCounter++;

  if (secondsCounter < SGP4_UPDATE_SECONDS)  // Respect the specified update interval
    return;
  secondsCounter = 0;

  // Daylight Saving Time is a coding nightmare.
  // The following performs a table-based check on United States DST.
  // The tabled data reflect US law as of the end of DST in year 2023.
  // The law can change at any time. To disable this check, undefine the symbol US_DST_TABLE
  // in the #include file DST.c. After 2037, Unix Epoch Time is problematic!
  // Finally, note that without this check the station parameter for DST is used and ...
  // it is therefore necessary to edit the station file whenever DST begins or ends.
  // Otherwise the displayed local time of pass predictions will be off by 1 hour.

  checkDST = verifyDST();
  if (checkDST >= 0)
    verifiedDSThours *= checkDST;

  // now() is local time, including DST if applicable
  unixTime = now() - (3600 * (int(fTZ) + verifiedDSThours));

  Serial.println();
  Serial.print("Unix time: ");
  Serial.println(unixTime);

  // One line satellite header, displayed whether pass in progress or not
  sat.findsat(unixTime);
  satCurrentRange = (float)sat.satDist;
  satCurrentRangeTimestamp = millis();
  satAz = String(sat.satAz);
  satAz = satAz.substring(0, satAz.length() - 1);  // Azimuth to 1 decimal digit
  satCurrentElv = sat.satEl;

  sDataLine = "Az ";
  sDataLine.concat(satAz);
  sDataLine.concat(" Range ");
  sDataLine.concat(satCurrentRange);
  tftDisplaySelectedSatelliteData(passAzRangeYindex, sDataLine);
  debugPrintSatAzElvRange();

  if (debugSimulatePassInProgress)
    pass_in_progress = true;

  if (pass_in_progress) {
    tftDisplayTxDopplerSendToggle();  // Debug (Will be in pass context)
    tftDisplayRxDopplerSendToggle();  // Debug (Will be in pass context)
  }

  // Velocity computation depends on two ranges and the time between them
  if (satLastRangeTimestamp > 0) {
    satRangeTimeDelta = SGP4_UPDATE_INTERVAL;  // SGP4 takes unix time (even seconds).
    if (satRangeTimeDelta > 0) {
      // Velocity relative to station lat/lon
      // Signed + means moving away, - means moving toward
      satRangeDelta = satCurrentRange - satLastRange;
      if (satRangeDelta < 0) {
        ascending = true;
        descending = false;
      } else if (satRangeDelta > 0) {
        ascending = false;
        descending = true;
      }
      satVelocity = satRangeDelta * 1000000. / (float)satRangeTimeDelta;
      // Velocity calculation complete
      debugPrintSatRangeVelocity();

      // Compute Doppler frequency adjustments - Invert Doppler sign for transmit.
      rxD = fDoppler((long)satVelocity, (float)receiveFrequencyVFO);
      if (invertTransmitDoppler)
        txD = fDoppler((long)satVelocity, (float)transmitFrequencyVFO);
      else
        txD = fDoppler((long)-satVelocity, (float)transmitFrequencyVFO);

      // Convert to long integer
      receiveFrequencyDopplerVFO = int(rxD);  // Update global values for rig interfaces
      transmitFrequencyDopplerVFO = int(txD);

      Serial.println("Before additive correction ...");
      debugPrintDopplerVFOs();

      if (transmitFrequencyLB != transmitFrequencyUB) {
        // If not single-frequency (e.g. FM) satellite,
        // respect additive correction.
        receiveFrequencyDopplerVFO += rxIntercept;
        transmitFrequencyDopplerVFO += txIntercept;
      }

      // Distinguish between pass imminent, pass in progress, or pass predicted
      // To do: Make 'pass imminent' status depend on time comparison.
      //        Satellite orbit can rise above -5 degrees, while not crossing observer's horizon
      if (satCurrentElv > -5. && satCurrentElv < 0 && ascending && !debugSimulatePassInProgress) {
        pass_in_progress = false;
      }

      // Display Doppler deltas in KHz
      rxDisplayDoppler = (rxD - (float)receiveFrequencyVFO) / 1000.;
      txDisplayDoppler = (txD - (float)transmitFrequencyVFO) / 1000.;

      if (!pass_in_progress && (satCurrentElv > 0.))
        initPositiveElevation();

      // Pass in progress
      if (satCurrentElv > 0. || debugSimulatePassInProgress) {  // Pass in progress
        passInProgress(txDisplayDoppler, rxDisplayDoppler);
      }

      if (satCurrentElv > -5. && satCurrentElv < 0 && descending && !debugSimulatePassInProgress) {
        // Pass has ended =OR= satellite ascended above -5 degrees bud did not cross 0 degrees.
        postPassCleanup();
      }

      debugPrintUncorrectedFrequencies();
      debugPrintDopplerCorrectedFrequencies();

      Serial.print("Rx Doppler: ");
      Serial.print(rxDisplayDoppler, 3);
      Serial.print(" kHz    Tx Doppler: ");
      Serial.print(txDisplayDoppler, 3);
      Serial.println(" kHz");
    }
  }  // End of in-pass or near-pass

  satLastRange = satCurrentRange;
  satLastRangeTimestamp = satCurrentRangeTimestamp;


  // Pass prediction section -
  int year;
  int mon;
  int day;
  int hr;
  int minute;
  double sec;
  boolean error;
  passinfo overpass;

  sat.initpredpoint(unixTime, 0.0);

  error = sat.nextpass(&overpass, 20);
  if (error == 0) {
    Serial.println("Error occurred while predicting passes.");
  } else {
    jd_of_predicted_pass = overpass.jdstart;
    ut_of_predicted_pass = j2u(jd_of_predicted_pass);
    Serial.print("jd_of_predicted_pass: ");
    Serial.println(jd_of_predicted_pass, 8);
    Serial.print("ut_of_predicted_pass: ");
    Serial.print(ut_of_predicted_pass);
    Serial.print("    Current Unix time: ");
    Serial.println(unixTime);
    // See note above for explanation of the following, where verified DST is folded into TZ.
    // invjday(...) computes the bounds of British Summer Time which differs from US DST.
    // DST will be specified in the station.txt parameters file, with optional verification.
    invjday(jd_of_predicted_pass, (int)fTZ + verifiedDSThours, false, year, mon, day, hr, minute, sec);
    Serial.println("Pass " + String(mon) + '/' + String(day) + '/' + String(year));
    Serial.println("  Start: az=" + String(overpass.azstart) + "° " + String(hr) + ':' + String(minute) + ':' + String(sec));
    // Save start-of-pass date/time before variables are recalculated below
    String startHr = String(hr);
    String startMin = String(minute);
    String startMon = String(mon);
    String startDay = String(day);
    String startYear = String(year);
    //
    invjday(overpass.jdmax, (int)fTZ + verifiedDSThours, false, year, mon, day, hr, minute, sec);
    satMaxElv = String(overpass.maxelevation);
    Serial.println("  Max: elev=" + satMaxElv + "° " + String(hr) + ':' + String(minute) + ':' + String(sec));
    invjday(overpass.jdstop, (int)fTZ + verifiedDSThours, false, year, mon, day, hr, minute, sec);
    Serial.println("  Stop: az=" + String(overpass.azstop) + "° " + String(hr) + ':' + String(minute) + ':' + String(sec));

    if (satCurrentElv < -5 && !debugSimulatePassInProgress) {  // Show next pass info on satellite data screen
      sDataLine = "Next pass max elv ";
      sDataLine.concat(satMaxElv.substring(0, satMaxElv.indexOf(POINT) + 2));
      tftDisplaySelectedSatelliteData(predictedPassMaxElvYindex, sDataLine);
      sDataLine = "at ";
      sDataLine.concat(startHr + ':');
      if (startMin.length() < 2)
        sDataLine.concat('0');
      sDataLine.concat(startMin);
      sDataLine.concat(" on ");
      sDataLine.concat(startMon + '/' + startDay + '/' + startYear + POINT);
      tftDisplaySelectedSatelliteData(predictedPassDateTimeYindex, sDataLine);
      pass_in_progress = false;
    }
  }

  //lastSatelliteInfoUpdate = millis();
}

// Encapsulate pass components.

void predictNextPass() {
  ;
}

void prePassPreparation() {
  ;
}

void initPositiveElevation() {
  // First iteration of update after start of pass
  // Erasures
  tftDisplaySelectedSatelliteData(predictedPassMaxElvYindex, "");
  tftDisplaySelectedSatelliteData(predictedPassDateTimeYindex, "");
}

void passInProgress(float txDisplayDoppler, float rxDisplayDoppler) {
  String sDataLine = "";
  // Radio interface
  sendReceiveFrequencyUpdate();
  sendTransmitFrequencyUpdate();
  if (lastSgp4Tick % ANT_INTERFACE_UPDATE_INTERVAL == 0)
    sendAntennaAzimuth();
  // Touch screen
  sDataLine = "Tx Doppler  Rx Doppler";
  tftDisplaySelectedSatelliteData(passDopplersCaptionYindex, sDataLine);
  sDataLine = "  ";
  sDataLine.concat(String(txDisplayDoppler));
  sDataLine.concat("        ");
  sDataLine.concat(String(rxDisplayDoppler));
  tftDisplaySelectedSatelliteData(passDopplersDataYindex, sDataLine);
  tftDisplayMultiwayInterfaceButtons(lockUnplugSwitchesYindex);
  pass_in_progress = true;
}

void postPassCleanup() {
  // Pass not in progress
  txDopplerCorrectionEnabled = false;
  rxDopplerCorrectionEnabled = false;
  tftEraseDiButtons(lockUnplugSwitchesYindex);
  pass_in_progress = false;
  // Pass just ended - Do once
  tftEraseSatelliteDataLine(passDopplersCaptionYindex);
  tftEraseSatelliteDataLine(passDopplersDataYindex);
  // Kludge try (half-button erase failure problem)
  tftEraseFullLengthButton(txDopplerSwitchYindex, satData_background_color);
  tftEraseFullLengthButton(rxDopplerSwitchYindex, satData_background_color);
  // Next - The full length erase covers both lock/unplug buttons
  tftEraseFullLengthButton(lockUnplugSwitchesYindex, satData_background_color);
}

// -----------------------------Miscellanous Utilities-----------------------------

void myDelay(unsigned long ms) {
  unsigned long endDelay = millis() + ms;
  while (millis() < endDelay) {
    if (ts.touched())
      return;
    delay(MILLISEC);
  }
  return;
}


// ------------------------------------Time----------------------------------------

void adjustEditableTime() {
  int buttonID = tftTimeAdjustButtonTouched();
  if (buttonID == 0)
    return;
  if (buttonID < 20) {
    if (buttonID == 16)
      deltaSecond++;
    else if (buttonID == 15)
      deltaMinute++;
    else if (buttonID == 14)
      deltaHour++;
    else if (buttonID == 12)
      deltaDay++;
    else if (buttonID == 11)
      deltaMonth++;
    else if (buttonID == 13)
      deltaYear++;
  } else {
    if (buttonID == 26)
      deltaSecond--;
    else if (buttonID == 25)
      deltaMinute--;
    else if (buttonID == 24)
      deltaHour--;
    else if (buttonID == 22)
      deltaDay--;
    else if (buttonID == 21)
      deltaMonth--;
    else if (buttonID == 23)
      deltaYear--;
  }
  tftDisplayCancelSaveButtons(cancelSaveButtonsYindex);
}

void adjustRTCtime() {
  time_t newTime;
  int buttonID = timeAdjustCancelSaveButtonTouched(cancelSaveButtonsYindex);
  if (buttonID == 0)
    return;
  if (buttonID == 1) {
    zeroDeltaDateTime();  // Noted separately for clarity
  } else if (buttonID == 2) {
    newTime = dmyhms2time(hour() + deltaHour, minute() + deltaMinute, second() + deltaSecond, day() + deltaDay, month() + deltaMonth, year() + deltaYear);
    setTime(newTime);
    Teensy3Clock.set(newTime);
    zeroDeltaDateTime();
  }
  tftEraseDiButtons(cancelSaveButtonsYindex);
  return;
}

unsigned long processSyncMessage() {
  unsigned long pctime = 0L;
  // Jan 1 2013
  const unsigned long DEFAULT_TIME = 1357041600;
  if (Serial.find(TIME_HEADER)) {
    pctime = Serial.parseInt();
    return pctime;
    if (pctime < DEFAULT_TIME) {  // Check for valid time (> Jan 1 2013)
      pctime = 0L;                // Error return 0
    }
  }
  return pctime;
}

void getTimeFromHost() {
  setSyncProvider(getTeensyTime);
  if (Serial.available()) {
    time_t t = processSyncMessage();
    if (t != 0) {
      t += HOST_TIME_DELTA;
      Teensy3Clock.set(t);
      setTime(t);
    }
  }
}

time_t getTeensyTime() {
  return Teensy3Clock.get();
}

time_t dmyhms2time(uint8_t h, uint8_t m, uint8_t s, uint8_t d, uint8_t mn, uint8_t y) {
  // Paraphrases TimeLib.h makeTime(const tmElements_t &tm)
  // Year is 2 or 4 digit year, as in TimeLib.h setTime(...)
  if (y > 99)  // Convert to Unix year
    y = y - 1970;
  else
    y += 30;

  // Remainder is from TimeLib.h makeTime
  int i;
  uint32_t seconds;

  // seconds from 1970 till 1 jan 00:00:00 of the given year
  seconds = y * (SECS_PER_DAY * 365);
  for (i = 0; i < y; i++) {
    if (LEAP_YEAR(i)) {
      seconds += SECS_PER_DAY;  // add extra days for leap years
    }
  }

  // add days for this year, months start from 1
  for (i = 1; i < mn; i++) {
    if ((i == 2) && LEAP_YEAR(y)) {
      seconds += SECS_PER_DAY * 29;
    } else {
      seconds += SECS_PER_DAY * monthDays[i - 1];  //monthDay array starts from 0
    }
  }
  seconds += (d - 1) * SECS_PER_DAY;
  seconds += h * SECS_PER_HOUR;
  seconds += m * SECS_PER_MIN;
  seconds += s;
  return (time_t)seconds;
}

void initClock() {
  time_t t = getTeensyTime();
  if (t != 0) {
    Teensy3Clock.set(t);
    setTime(t);
  }
}

String sTime() {
  String t = "";
  if (hour() < 10)
    t.concat("0");
  t.concat(hour());
  t.concat(":");
  if (minute() < 10)
    t.concat("0");
  t.concat(minute());
  t.concat(":");
  if (second() < 10)
    t.concat("0");
  t.concat(second());
  return t;
}

// Time set / adjust

void zeroDeltaDateTime() {
  deltaYear = 0;   // All deltas are signed
  deltaMonth = 0;  // RTC value + delta = True (correct) value
  deltaDay = 0;
  deltaHour = 0;  // 24 hour format
  deltaMinute = 0;
  deltaSecond = 0;
}

void tftDisplayEditableDateTimeRow(int yNdx, uint16_t txtC, uint16_t sz) {
  String s = "m";
  int mo = month() + deltaMonth;
  int d = day() + deltaDay;
  int y = year() + deltaYear;
  int h = hour() + deltaHour;
  int mi = minute() + deltaMinute;
  int sec = second() + deltaSecond;
  int iTmp = (mo + 12) % 12;
  if (iTmp == 0)  // Months are numbered 1 to 12 (no zero month)
    iTmp = 12;
  if (iTmp < 10)
    s.concat(ZERO);
  s.concat(String(iTmp));
  s.concat(" d");  // No format enforcement for day or year
  if (d < 10)
    s.concat(ZERO);
  s.concat(String(d));
  s.concat(" ");
  s.concat(String(y));
  s.concat(" h");  // Format hours, minutes, seconds display, 0 - 23 (hrs) 0 - 59 (m/s)
  iTmp = (h + 24) % 24;
  if (iTmp < 10)
    s.concat(ZERO);
  s.concat(String(iTmp));
  s.concat(" m");
  iTmp = (mi + 60) % 60;
  if (iTmp < 10)
    s.concat(ZERO);
  s.concat(String(iTmp));
  s.concat(" s");
  iTmp = (sec + 60) % 60;
  if (iTmp < 10)
    s.concat(ZERO);
  s.concat(String(iTmp));
  tft.setTextColor(txtC);
  tft.setTextSize(sz);
  tft.fillRect(load_button_x0, button_y0 + yNdx * button_height, load_button_width, button_height, toolscreen_background_color);
  tft.setCursor(load_button_x0, button_y0 + yNdx * button_height);
  tft.print(s);
}

int verifyDST() {  // US daylight saving time
#ifndef US_DST_TABLE
  return -1;
#endif
  // Return 1 iff current epoch time is within United States Daylight Saving Time dates for the year.
  // Return 0 iff current epoch time is outside US DST dates for the year.
  // Return -1 if DST status cannot be ascertained, due to year out-of-bounds or any other reason.
  int yearNdx = year() - START_DST_YEAR;
  if (!(yearNdx < NUM_DST_YEARS))
    return -1;
  time_t t = getTeensyTime();
  if (t == 0)
    return -1;
  if (DST_DT[yearNdx][0] < t && t < DST_DT[yearNdx][1])
    return 1;
  else
    return 0;
}

// -----------------------------------EEPROM---------------------------------------

//  Adapted from MOAK-IV and previous projects
void writeEEPROM(String s) {
  int sLen = s.length();
  if (sLen > EEPROM.length())
    return;
  for (int i = 0; i < sLen; i++)
    EEPROM.write(i, s.charAt(i));
}

void writeEEPROM(String s, int iPos) {
  int sLen = s.length();
  if (sLen > EEPROM.length())
    return;
  for (int i = 0; i < sLen; i++)
    EEPROM.write(i + iPos, s.charAt(i));
}

String readEEPROM(int sLen) {
  if (sLen > EEPROM.length()) sLen = EEPROM.length();
  String s = "";
  for (int i = 0; i < sLen; i++)
    s.concat((char)EEPROM.read(i));
  return s;
}

String readEEPROM(int sLen, int iPos) {
  if (sLen > EEPROM.length()) sLen = EEPROM.length();
  String s = "";
  for (int i = 0; i < sLen; i++)
    s.concat((char)EEPROM.read(i + iPos));
  return s;
}

void eraseEEPROM(int len) {
  if (len > EEPROM.length()) len = EEPROM.length();
  for (int i = 0; i < len; i++)
    EEPROM.write(i, 0);
}

void eeSaveIntercepts() {
  String s = EEDLM;
  s.concat(String(txIntercept));
  while (s.length() < 7)
    s.concat(SP);
  s.concat(EEDLM);
  s.concat(String(rxIntercept));
  while (s.length() < 14)
    s.concat(SP);
  s.concat(EEDLM);
  writeEEPROM(s);
}

void eeRestoreIntercepts() {
  if (readEEPROM(1, EE_START) == EEDLM) {
    // Validity check OK
    txIntercept = readEEPROM(6, EE_START + 1).toInt();
    rxIntercept = readEEPROM(6, EE_START + 8).toInt();
  }
}

void eeStartupRestore() {
  eeRestoreIntercepts();
}

// --------------------------------My Screensaver---------------------------------

boolean screensaverOn() {
  if (millis() - lastTouched > SCREEN_TIMEOUT && (screenID != 2)) {
    tft.fillScreen(ssBkgColor);
    return true;
  }
  return false;  
}

// LM: See next section (Imported/Adapted) for 'Bubbles' screensaver

void oneSatScreenSaverIteration(unsigned long glyphDuration) {
  // Satellite glyph
  const uint16_t LEFT_BOUND = 0;              // x (Computable from satellite dimensions)
  const uint16_t RIGHT_BOUND = 235;
  const uint16_t UPPER_BOUND = 8;             // y (Ditto)
  const uint16_t LOWER_BOUND = 177;
  const int NUMBER_OF_COLORS = 4;
  uint16_t x, y;
  uint16_t panelColor;
  int fillColor = random(NUMBER_OF_COLORS);
  unsigned long startGlyphDisplay = millis();
  x = random(RIGHT_BOUND-LEFT_BOUND+1);
  y = random(LOWER_BOUND-UPPER_BOUND+1) + UPPER_BOUND;
  if (fillColor == 0)
    panelColor = YELLOW;
  else if (fillColor == 1)
    panelColor = BLUE;
  else if (fillColor == 2)
    panelColor = RED;
  else //if (fillColor == 3)
    panelColor = GREEN;
  tftDrawSatelliteGlyph(x, y, WHITE, panelColor);
  while (startGlyphDisplay + glyphDuration > millis()) {
    if (tftAbort())
      return;
      delay(MILLISEC);
  }
  // Erase glyph
  tftDrawSatelliteGlyph(x, y, ssBkgColor, ssBkgColor);
}

// --------------------------------Math & Utility---------------------------------

float radius(float chordLength, float segHeight) {
  float lsqr = chordLength*chordLength;
  float hsqr = segHeight*segHeight;
  return (4 * hsqr + lsqr) / (8 * segHeight);
}

float theta(float chordLength, float radius) {
  return acos(chordLength/(radius+radius));   // Radians
}

float r2d(float radians) {
  return radians * 57.2957795;                // Degrees
}

// -------------------------------Imported/Adapted---------------------------------

// Next adapted from: https://afterthoughtsoftware.com/posts/convert-rgb888-to-rgb565
uint16_t uColor(uint8_t RGB[]) {

  uint16_t b = (RGB[2] >> 3) & 0x1f;
  uint16_t g = ((RGB[1] >> 2) & 0x3f) << 5;
  uint16_t r = ((RGB[0] >> 3) & 0x1f) << 11;

  return (uint16_t)(r | g | b);
}

// Next adapted from: https://stackoverflow.com/questions/466321/convert-unix-timestamp-to-julian
double u2j(unsigned long uSecs) {
  return (uSecs / 86400.0) + 2440587.5;
}

// Similarly from: https://www.php.net/manual/en/function.jdtounix.php
unsigned long j2u(double jd) {
  return (unsigned long)((jd - 2440587.5) * 86400.);
}

// Screensaver - This is [an adaptation of] the Arduino TFT bubbles demo sketch
//               The following comment is reproduced from the example sketch

/*
	This example was adapted from ugfx http://ugfx.org
	It's a great example of many 2d objects in a 3d space (matrix transformations)
	and show[s] the capabilities of RA8875 chip.
 Tested and worked with:
 Teensy3, Teensy3.1, Arduino UNO, Arduino YUN, Arduino Leonardo, Stellaris
 Works with Arduino 1.0.6 IDE, Arduino 1.5.8 IDE, Energia 0013 IDE
*/

#define NDOTS 512			// Number of dots 512
#define SCALE 4096//4096
#define INCREMENT 512//512
#define PI2 6.283185307179586476925286766559
#define RED_COLORS (32)
#define GREEN_COLORS (64)
#define BLUE_COLORS (32)

int16_t sine[SCALE+(SCALE/4)];
int16_t *cosi = &sine[SCALE/4];
int16_t angleX = 0, angleY = 0, angleZ = 0;
int16_t speedX = 0, speedY = 0, speedZ = 0;

int16_t xyz[3][NDOTS];
uint16_t col[NDOTS];
int pass = 0;

void initScreenSaver (void){
  uint16_t i;
  /* if you change the SCALE*1.25 back to SCALE, the program will
   * occassionally overrun the cosi array -- however this actually
   * produces some interesting effects as the BUBBLES LOSE CONTROL!!!!
   */
  for (i = 0; i < SCALE+(SCALE/4); i++)
    //sine[i] = (-SCALE/2) + (int)(sinf(PI2 * i / SCALE) * sinf(PI2 * i / SCALE) * SCALE);
    sine[i] = (int)(sinf(PI2 * i / SCALE) * SCALE);
}

void matrix (int16_t xyz[3][NDOTS], uint16_t col[NDOTS]){
  static uint32_t t = 0;
  int16_t x = -SCALE, y = -SCALE;
  uint16_t i, s, d;
  uint8_t red,grn,blu;

  for (i = 0; i < NDOTS; i++)
  {
/*
    if (tftAbort()) {
      Serial.println("tftAbort() @ 1");
      return;
    }
*/
    xyz[0][i] = x;
    xyz[1][i] = y;

    d = sqrt(x * x + y * y); 	/* originally a fastsqrt() call */
    s = sine[(t * 30) % SCALE] + SCALE;

    xyz[2][i] = sine[(d + s) % SCALE] * sine[(t * 10) % SCALE] / SCALE / 2;

    red = (cosi[xyz[2][i] + SCALE / 2] + SCALE) * (RED_COLORS - 1) / SCALE / 2;
    grn = (cosi[(xyz[2][i] + SCALE / 2 + 2 * SCALE / 3) % SCALE] + SCALE) * (GREEN_COLORS - 1) / SCALE / 2;
    blu = (cosi[(xyz[2][i] + SCALE / 2 + SCALE / 3) % SCALE] + SCALE) * (BLUE_COLORS - 1) / SCALE / 2;
    col[i] = ((red << 11) + (grn << 5) + blu);
    x += INCREMENT;
    if (x >= SCALE) x = -SCALE, y += INCREMENT;
  }
  t++;
}

void rotate (int16_t xyz[3][NDOTS], uint16_t angleX, uint16_t angleY, uint16_t angleZ){
  uint16_t i;
  int16_t tmpX, tmpY;
  int16_t sinx = sine[angleX], cosx = cosi[angleX];
  int16_t siny = sine[angleY], cosy = cosi[angleY];
  int16_t sinz = sine[angleZ], cosz = cosi[angleZ];

  for (i = 0; i < NDOTS; i++)
  {
/*
    if (tftAbort()) {
      Serial.println("tftAbort() @ 2");
      return;
    }
*/
    tmpX      = (xyz[0][i] * cosx - xyz[2][i] * sinx) / SCALE;
    xyz[2][i] = (xyz[0][i] * sinx + xyz[2][i] * cosx) / SCALE;
    xyz[0][i] = tmpX;
    tmpY      = (xyz[1][i] * cosy - xyz[2][i] * siny) / SCALE;
    xyz[2][i] = (xyz[1][i] * siny + xyz[2][i] * cosy) / SCALE;
    xyz[1][i] = tmpY;
    tmpX      = (xyz[0][i] * cosz - xyz[1][i] * sinz) / SCALE;
    xyz[1][i] = (xyz[0][i] * sinz + xyz[1][i] * cosz) / SCALE;
    xyz[0][i] = tmpX;
  }
}

void draw(int16_t xyz[3][NDOTS], uint16_t col[NDOTS]){
  static uint16_t oldProjX[NDOTS] = { 0 };
  static uint16_t oldProjY[NDOTS] = { 0 };
  static uint8_t oldDotSize[NDOTS] = { 0 };
  uint16_t i, projX, projY, projZ, dotSize;

  for (i = 0; i < NDOTS; i++)
  {
    if (tftAbort()) {
//    Serial.println("tftAbort() @ 3");
      return;
    }
    projZ = SCALE - (xyz[2][i] + SCALE) / 4;
    projX = tft.width() / 2 + (xyz[0][i] * projZ / SCALE) / 25;
    projY = tft.height() / 2 + (xyz[1][i] * projZ / SCALE) / 25;
    dotSize = 3 - (xyz[2][i] + SCALE) * 2 / SCALE;

    tft.drawCircle (oldProjX[i], oldProjY[i], oldDotSize[i], BLACK);

    if (projX > dotSize && projY > dotSize && projX < tft.width() - dotSize && projY < tft.height() - dotSize)
    {
      tft.drawCircle (projX, projY, dotSize, col[i]);
      oldProjX[i] = projX;
      oldProjY[i] = projY;
      oldDotSize[i] = dotSize;
    }
  }
}

void oneBubblesScreenSaverIteration() 
{
  matrix(xyz, col);
  rotate(xyz, angleX, angleY, angleZ);
  draw(xyz, col);
  angleX += speedX;
  angleY += speedY;
  angleZ += speedZ;

  if (pass > 400) speedY = 1;
  if (pass > 800) speedX = 1;
  if (pass > 1200) speedZ = 1;
  pass++;

  if (angleX >= SCALE) {
    angleX -= SCALE;
  } 
  else if (angleX < 0) {
    angleX += SCALE;
  }

  if (angleY >= SCALE) {
    angleY -= SCALE;
  } 
  else if (angleY < 0) {
    angleY += SCALE;
  }

  if (angleZ >= SCALE) {
    angleZ -= SCALE;
  } 
  else if (angleZ < 0) {
    angleZ += SCALE;
  }
}


// ----------------------------------Debug/Test------------------------------------

void printFreqList() {
  for (int i = 0; i < satCount; i++) {
    Serial.println(freqList[i]);
  }
}

void printSelectedTLE() {
  Serial.println();
  Serial.println(currentTLE[0]);
  Serial.println(currentTLE[1]);
}

void printStationParameters() {
  Serial.println();
  Serial.println("Station parameters:");
  Serial.println(sCallsign);
  Serial.println(fLat, 4);
  Serial.println(fLon, 4);
  Serial.println(fTZ);
  Serial.println(DST);
  Serial.println(xmtInterface);
  Serial.println(rcvInterface);
  Serial.println(xmtType);
  Serial.println(rcvType);
  Serial.println(xmtBAUD);
  Serial.println(rcvBAUD);
  Serial.println(tuningPreference);
  Serial.println(DopplerPreference);
  Serial.println();
}

void printSatNames() {
  Serial.println();
  Serial.print("satNames[] array contents");
  for (int i = 0; i < MAX_SATS; i++)
    Serial.println(satNames[i]);
}

void debugPrintSatAzElvRange() {
  Serial.print("Azimuth: " + satAz);
  Serial.print("    Elevation = " + String(satCurrentElv));
  Serial.print("    Range: " + String(satCurrentRange));
  Serial.println(" km");
}

void debugPrintSatRangeVelocity() {
  Serial.print("satRangeDelta: ");
  Serial.print(satRangeDelta, 4);
  Serial.print(" km    timeDelta: ");
  Serial.print(satRangeTimeDelta);
  Serial.println(" ms");
  Serial.print("Computed velocity: ");
  Serial.print(satVelocity);
  Serial.println(" mtrs/sec");
  Serial.print("Timestamps (ms) Current: ");
  Serial.print(satCurrentRangeTimestamp);
  Serial.print(", Previous: ");
  Serial.println(satLastRangeTimestamp);
}

void debugPrintDopplerVFOs() {
  Serial.print("Rx Doppler VFO: ");
  Serial.print(receiveFrequencyDopplerVFO);
  Serial.print("  Tx Doppler VFO: ");
  Serial.println(transmitFrequencyDopplerVFO);
}

void debugPrintUncorrectedFrequencies() {
  Serial.print("Uncorrected (tuned) receive frequency: ");
  Serial.print(receiveFrequencyVFO);
  Serial.print(" Hz   Uncorrected (tuned) transmit frequency: ");
  Serial.print(transmitFrequencyVFO);
  Serial.println(" Hz");
}

void debugPrintDopplerCorrectedFrequencies() {
  Serial.print("Doppler corrected receive frequency: ");
  Serial.print(receiveFrequencyDopplerVFO);
  Serial.print(" Hz   Doppler corrected transmit frequency: ");
  Serial.print(transmitFrequencyDopplerVFO);
  Serial.println(" Hz");
}


void testSGP4() {
  // From Sparkfun's example Sgp4Tracker.ino
  //Display TLE epoch time
  int year;
  int mon;
  int day;
  int hr;
  int minute;
  double sec;
  double jdC = sat.satrec.jdsatepoch;
  invjday(jdC, DST_hours + (int)fTZ, true, year, mon, day, hr, minute, sec);
  // Inverting date format (Example was European format)
  Serial.println("Satellite: " + satNames[selectedSatellite]);
  Serial.println("Epoch: " + String(mon) + '/' + String(day) + '/' + String(year) + ' ' + String(hr) + ':' + String(minute) + ':' + String(sec));
  Serial.println("Day/hour has been converted to local time. Set fTZ = 0. for Universal Time.");
  Serial.println();
}

void displayLoopbackData() {
  String s;
  for (int i = 0; i < rxNdx; i++)
    s.concat((char)serRxBuffer[i]);
  displayLCD("Received:", s);
}


void civTest() {
  // Called from the end of setup() followed by a delay, if needed
  char c;
  unsigned long testFreq = 145950123;
  cat2civFreqFormat(String(testFreq));
  Serial.print("Test frequency: ");
  Serial.print(testFreq);
  Serial.print(" to CI-V format: ");
  Serial.println(CIV_Frequency);
  //
  String testCommand = civFreqCommand(CIV_Frequency);
  unsigned int L = testCommand.length();
  Serial.print("Test command length: ");
  Serial.println(L);
  Serial.print("Test command: ");
  for (unsigned int i = 0; i < L; i++) {
    c = testCommand.charAt(i);
    if (c >= '0' && c <= '9') {
      Serial.print(c);
      Serial.print(SP);
    } else {
      Serial.print("0x");
      Serial.print((byte)c < 16 ? "0" : "");
      Serial.print((byte)c, HEX);
      Serial.print(SP);
    }
    //
  }
  Serial.println();
  //
  // To test the actual transmitted message requires simulating satellite context variables
  // Store and restore these
  boolean prevXmtInterface = xmtInterface;
  String prevXmtType = xmtType;
  boolean prevTxDopplerCorrectionEnabled = txDopplerCorrectionEnabled;
  unsigned long prevTransmitFrequencyVFO = transmitFrequencyVFO;
  //
  xmtInterface = true;
  xmtType = "88h";
  txDopplerCorrectionEnabled = false;
  transmitFrequencyVFO = testFreq;
  //
  Serial.print("Transmitted: ");
  sendTransmitFrequencyUpdate();  // Temporarily modified to print transmitted data
  Serial.println();
  //
  transmitFrequencyVFO = prevTransmitFrequencyVFO;
  txDopplerCorrectionEnabled = prevTxDopplerCorrectionEnabled;
  xmtType = prevXmtType;
  xmtInterface = prevXmtInterface;
}

void multSendTest() {
  unsigned long f = 145950000;
  for (unsigned int i = 0; i < 10; i++) {
    civTest(f + (unsigned long)random(1000));
    delay(MOMENT);
    for (unsigned int j = 0; j < 1000; j++) {
      rigInterface0Listen();
      if (rigCommandReceived) {
        processReceivedCivMessage();
        zeroRxBuffer();
      }
    }
    delay(ONESEC);
  }
}

void civTest(unsigned long testFreq) {
  // Called from the end of setup() followed by a delay, if needed
  char c;
  cat2civFreqFormat(String(testFreq));
  Serial.print("Test frequency: ");
  Serial.println(testFreq);
  //
  String testCommand = civFreqCommand(CIV_Frequency);
  unsigned int L = testCommand.length();
  Serial.print("Test command length: ");
  Serial.println(L);
  Serial.print("Test command: ");
  for (unsigned int i = 0; i < L; i++) {
    c = testCommand.charAt(i);
    if (c >= '0' && c <= '9') {
      Serial.print(c);
      Serial.print(SP);
    } else {
      Serial.print("0x");
      Serial.print((byte)c < 16 ? "0" : "");
      Serial.print((byte)c, HEX);
      Serial.print(SP);
    }
    //
  }
  Serial.println();
  //
  // To test the actual transmitted message requires simulating satellite context variables
  // Store and restore these
  boolean prevXmtInterface = xmtInterface;
  String prevXmtType = xmtType;
  boolean prevTxDopplerCorrectionEnabled = txDopplerCorrectionEnabled;
  unsigned long prevTransmitFrequencyVFO = transmitFrequencyVFO;
  //
  xmtInterface = true;
  xmtType = "88h";
  txDopplerCorrectionEnabled = false;
  transmitFrequencyVFO = testFreq;
  //
  Serial.print("Transmitted: ");
  sendTransmitFrequencyUpdate();  // Temporarily modified to print transmitted data
  Serial.println();
  //
  transmitFrequencyVFO = prevTransmitFrequencyVFO;
  txDopplerCorrectionEnabled = prevTxDopplerCorrectionEnabled;
  xmtType = prevXmtType;
  xmtInterface = prevXmtInterface;
}

void multCivRequestTest() {
  for (unsigned int i = 0; i < 5; i++) {
    civRequestTest();
    delay(MOMENT);
    for (unsigned int j = 0; j < 1000; j++) {
      rigInterface0Listen();
      if (rigCommandReceived) {
        processReceivedCivMessage();
        zeroRxBuffer();
      }
    }
    delay(ONESEC);
  }
}

void civRequestTest() {
  const int MSG_LENGTH = 6;
  byte requestOperatingFrequencyMsg[MSG_LENGTH] = { 0xFE, 0xFE, 0x88, 0xE0, 0x03, 0xFD };
  for (unsigned int i = 0; i < MSG_LENGTH; i++)
    rigInterface0.write(requestOperatingFrequencyMsg[i]);
  //rigInterface0.flush();
}

void multTestIC7300() {
  for (unsigned int i = 0; i < 5; i++) {
    testIC7300();
    delay(MOMENT);
    for (unsigned int j = 0; j < 1000; j++) {
      rigInterface0Listen();
      if (rigCommandReceived) {
        processReceivedCivMessage();
        zeroRxBuffer();
      }
    }
    delay(ONESEC);
  }
}

void testIC7300() {
  Serial.println("Sending request for operating frequency message");
  Serial.print("Received: ");
  const int MSG_LENGTH = 6;
  byte requestOperatingFrequencyMsg[MSG_LENGTH] = { 0xFE, 0xFE, 0x94, 0xE0, 0x03, 0xFD };
  for (unsigned int i = 0; i < MSG_LENGTH; i++)
    rigInterface0.write(requestOperatingFrequencyMsg[i]);
  //rigInterface0.flush();
  Serial.println();
}

void testTTL2RS232c() {
  String s = "This is rigInterface ";
  const byte CR = 13;
  const byte ZERO = 48;
  const byte ONE = 49;
  for (int i = 0; i < 10; i++) {
    if (ts.touched())  // This is a no-op!
      return;
    for (unsigned int j = 0; j < s.length(); j++) {
      rigInterface0.write((byte)s.charAt(j));
    }
    rigInterface0.write(ZERO);
    rigInterface0.write(CR);
    //
    for (unsigned int j = 0; j < s.length(); j++) {
      rigInterface1.write((byte)s.charAt(j));
    }
    rigInterface1.write(ONE);
    rigInterface1.write(CR);
    delay(TWOSEC);
  }
}

void testSendAz() {
  String s;
  Serial.print("Sending test message: ");
  Serial.println(testAntInterface);
  for (unsigned int i = 0; i < testAntInterface.length(); i++)
    antInterface.write(testAntInterface.charAt(i));
  antInterface.write(CR);
  Serial.print("Sending integer degrees: ");
  for (unsigned int n = 0; n < 10; n++) {
    s = "AZ";
    s.concat(String(random(360)));
    Serial.println(s);
    for (unsigned int i = 0; i < s.length(); i++)
      antInterface.write(s.charAt(i));
    antInterface.write(CR);
  }
  Serial.print("Sending decimal degrees: ");
  for (unsigned int n = 0; n < 10; n++) {
    s = "AZ";
    s.concat(String(random(360)));
    s.concat(POINT);
    s.concat(String(random(10)));
    Serial.println(s);
    for (unsigned int i = 0; i < s.length(); i++)
      antInterface.write(s.charAt(i));
    antInterface.write(CR);
  }
}

void testVectorGraphic() {
  // Exersize shapes supported by Adafruit GFX library
  // Invoked from tools screen
  tft.fillScreen(toolscreen_background_color);
  // Test code here
  for (unsigned int i=0; i<10; i++) {
    oneSatScreenSaverIteration(TWOSEC);
  }
  loadToolsScreen();
}

void toggleTestCondition(int testIndex) {
  // Index is zero-based
  if (0 <= testIndex && testIndex < MAX_TEST_CONDITIONS)
    test_condition[testIndex] = !test_condition[testIndex];
}

void processTestButtons() {
  int buttonID = tftTestButtonTouched();
  toggleTestCondition(buttonID);
  if (buttonID == 0) {
    if (test_condition[0] == true) {
      debugSimulatePassInProgress = true;
      statusMessageNumber = 1;
    } else {
      debugSimulatePassInProgress = false;
      statusMessageNumber = -1;
    }
  } else if (buttonID == 1) {
    if (test_condition[1]) {
      calibrationToolEnabled = true;
      tftDisplayFrequencyCalibrationComponents();
    } else {
      calibrationToolEnabled = false;
      tftEraseFrequencyCalibrationComponents();
    }
  }
  else if (buttonID == 2) {                   // Deprecated
/*
    if (testAzSerialInterface) {
      testAzSerialInterface = false;
      statusMessageNumber = -1;
    } else {
      testAzSerialInterface = true;
      statusMessageNumber = 2;
    }
*/
  }
  else if (buttonID == 3) {                   // Temporarily disabled  
    testVectorGraphic();
    toggleTestCondition(3);
  }
  else if (buttonID == 5) {                   // Deprecated
/*
    if (test_condition[5]) {
      invertTransmitDoppler = true;
      statusMessageNumber = 0;
    }
    else {
      invertTransmitDoppler = false;
      statusMessageNumber = -1;
    }
*/
  }
  tftDisplayTestButtons(testButtonsYindex);  // Refresh for color change
}

void processDebugSwitch() {
  // Unconditional debug here
  if (digitalRead(DEBUG_SWITCH) == HIGH)
    return;
  // DIP-switch enabled debug below
  displayTestButtons = true;
}
